import logging

from infra.callisto.controllers.sdk import tier
from infra.callisto.controllers.user.jupiter.chunks import allocator
from infra.callisto.controllers.user.jupiter.chunks import configs_erasure
from infra.callisto.controllers.user.jupiter import tier1 as web_tier_chunks
from infra.callisto.controllers.user.img import tier_chunks as images_tier_chunks
from infra.callisto.controllers.user.video import tier_chunks as video_tier_chunks
from infra.callisto.controllers.utils import gencfg_api
from infra.callisto.controllers.utils import funcs

from search.config.idl.service_config_pb2 import TDNSCache  # noqa


def generate_configs(generation, location, topology,
                     remote_storage_group, basesearch_groups, basesearch_hamster_groups,
                     namespace_prefix, chunk_namespace_prefix, subresources,
                     tier_name=tier.WebTier1.name,
                     endpoint_set_template=None, endpoint_set_hamster_template=None,
                     timeouts_multiplicator=1):
    logging.info('Start generation %s at location %s', generation, location)

    # Make protobuf happy.
    location = str(location)
    topology = str(topology)
    namespace_prefix = str(namespace_prefix)
    subresources = map(str, subresources)

    remote_storage_group = gencfg_api.GencfgGroup(remote_storage_group, topology, mtn=True)
    remote_storage_host_agent_map = gencfg_api.get_strict_host_agent_mapping([remote_storage_group])
    logging.info('Remote storage: %s, %s hosts', remote_storage_group, len(remote_storage_host_agent_map))

    if basesearch_groups:
        basesearch_groups = [gencfg_api.GencfgGroup(group, topology, mtn=True)
                             for group in basesearch_groups]
        basesearch_agent_shard_number_map = gencfg_api.get_agent_shard_number_mapping(basesearch_groups)
    else:
        basesearch_agent_shard_number_map = {}

    logging.info('Basesearch: %s, %s hosts',
                 basesearch_groups, len(basesearch_agent_shard_number_map))

    if basesearch_hamster_groups:
        basesearch_hamster_groups = [gencfg_api.GencfgGroup(group, topology, mtn=True)
                                     for group in basesearch_hamster_groups]

        basesearch_hamster_agent_shard_number_map = gencfg_api.get_agent_shard_number_mapping(basesearch_hamster_groups)
    else:
        basesearch_hamster_agent_shard_number_map = {}

    logging.info('Basesearch hamster: %s, %s hosts',
                 basesearch_hamster_groups, len(basesearch_hamster_agent_shard_number_map))

    alloc = allocator.Allocator(location, tier.TIERS[tier_name], chunk_namespace_prefix, readonly=True)
    logging.info('Allocator created for %s %s', alloc._location, alloc._tier)
    mapping = alloc.load_mapping(generation)
    logging.info('Load mapping %s: %s elements', generation, len(mapping))

    configs = configs_erasure.generate_configs(
        remote_storage_host_agent_map=remote_storage_host_agent_map,
        basesearch_agent_shard_number_map=basesearch_agent_shard_number_map,
        basesearch_hamster_agent_shard_number_map=basesearch_hamster_agent_shard_number_map,
        chunk_hosts_mapping=mapping,
        chunk_generator=make_chunk_generator(tier_name),
        namespace_prefix=namespace_prefix,
        subresources=subresources,
        endpoint_set_template=endpoint_set_template,
        endpoint_set_hamster_template=endpoint_set_hamster_template,
        timeouts_multiplicator=timeouts_multiplicator
    )

    config_files = {}
    config_files.update(_extract_instance_configs([remote_storage_group], configs))
    logging.info('Generate instance configs: %s elements', len(config_files))

    config_files.update(_extract_shardwise_configs(generation, basesearch_groups, configs))
    config_files.update(_extract_shardwise_configs(generation, basesearch_hamster_groups, configs, suffix='.hamster'))
    logging.info('Update with shardwise configs: %s elements', len(config_files))

    return config_files


def make_chunk_generator(tier_name):
    if tier_name.startswith('Web'):
        return web_tier_chunks.make_chunk_generator(tier.TIERS[tier_name], chunk_size_override=None)
    elif tier_name.startswith('Video'):
        return video_tier_chunks.make_chunk_generator(tier.TIERS[tier_name], chunk_size_override=None)
    elif tier_name.startswith('Img'):
        return images_tier_chunks.make_chunk_generator(tier.TIERS[tier_name], chunk_size_override=None)

    raise RuntimeError('Tier {} not supported'.format(tier_name))


def generate_dns_cache(topology, remote_storage_group, basesearch_groups, basesearch_hamster_groups=None):
    basesearch_groups.extend(basesearch_hamster_groups or [])

    dns_cache = TDNSCache()
    _append_dns_cache(dns_cache, remote_storage_group, topology)
    for basesearch_group in basesearch_groups:
        _append_dns_cache(dns_cache, basesearch_group, topology)

    logging.info('Generate dns cache: %s elements', len(dns_cache.Entities))
    return dns_cache


def _append_dns_cache(dns_cache, group, topology):
    for agent, instance in gencfg_api.searcher_lookup_agents(group, topology, mtn=True).iteritems():
        entity = dns_cache.Entities.add()
        entity.Host = agent.host
        entity.IpAddress = instance['hbf']['interfaces']['backbone']['ipv6addr']


def _extract_instance_configs(groups, configs):
    return {
        _make_config_name(agent): configs[agent]
        for agent in gencfg_api.get_agents(groups)
    }


def _extract_shardwise_configs(generation, basesearch_groups, configs, suffix=None):
    generation = str(generation)

    shardwise_configs = {}
    for agent, instance in gencfg_api.get_agents_instances(basesearch_groups).iteritems():
        shardwise_config_name = '{}.proto'.format(
            instance['shard_name']
        ).replace('0000000000', generation).replace('00000000-000000', funcs.timestamp_to_yt_state(generation))
        if suffix:
            shardwise_config_name += suffix
        if agent in configs:
            shardwise_configs.setdefault(shardwise_config_name, configs[agent])
    return shardwise_configs


def _make_config_name(agent):
    if 'gencfg-c' in agent.host:
        return '{}.proto'.format(agent.host)
    return '{}:{}.proto'.format(agent.host, agent.port)
