import os
import sys
import logging
import argparse

import infra.callisto.tools.rs_config_generator.lib as lib
import infra.callisto.controllers.user.jupiter.man as man
import infra.callisto.controllers.user.jupiter.vla as vla

import infra.callisto.controllers.user.img.sas as images_sas

import infra.callisto.controllers.user.video.sas as video_sas

from infra.callisto.controllers.utils import gencfg_api
from search.config.idl.service_config_pb2 import TDNSCache  # noqa


TIER_LOCATIONS_ENV = {
    'WebTier0': {
        'man': {
            'remote_storage_group': man.TIER0_REMOTE_STORAGE_GROUP,
            'basesearch_groups': [man.TIER0_BASESEARCH_GROUP, man.TIER0_ATTR_BASESEARCH_GROUP],
            'basesearch_hamster_groups': man.TIER0_BASESEARCH_HAMSTER_GROUPS,
            'endpoint_set_template': man.TIER0_ENDPOINT_SET_TEMPLATE,
            'endpoint_set_hamster_template': man.TIER0_ENDPOINT_SET_HAMSTER_TEMPLATE,
            'subresources': man.SUBRESOURCES
        },
        'vla': {
            'remote_storage_group': vla.TIER0_REMOTE_STORAGE_GROUP,
            'basesearch_groups': [vla.TIER0_BASESEARCH_GROUP, vla.TIER0_ATTR_BASESEARCH_GROUP],
            'basesearch_hamster_groups': vla.TIER0_BASESEARCH_HAMSTER_GROUPS,
            'endpoint_set_template': vla.TIER0_ENDPOINT_SET_TEMPLATE,
            'endpoint_set_hamster_template': vla.TIER0_ENDPOINT_SET_HAMSTER_TEMPLATE,
            'subresources': vla.SUBRESOURCES
        }
    },
    'VideoPlatinum': {
        'sas': {
            'remote_storage_group': video_sas.PLATINUM_REMOTE_STORAGE_GROUP,
            'basesearch_groups': [video_sas.PLATINUM_BASESEARCH_GROUP],
            'basesearch_hamster_groups': [],
            'endpoint_set_template': video_sas.PLATINUM_ENDPOINT_SET_TEMPLATE,
            'endpoint_set_hamster_template': None,
            'subresources': video_sas.SUBRESOURCES
        }
    },
    'ImgTier0': {
        'sas': {
            'remote_storage_group': images_sas.TIER0_REMOTE_STORAGE_GROUP,
            'basesearch_groups': [images_sas.TIER0_BASESEARCH_GROUP],
            'basesearch_hamster_groups': [images_sas.TIER0_BASESEARCH_HAMSTER_GROUP],
            'endpoint_set_template': images_sas.TIER0_ENDPOINT_SET_TEMPLATE,
            'endpoint_set_hamster_template': images_sas.TIER0_ENDPOINT_SET_HAMSTER_TEMPLATE,
            'subresources': images_sas.SUBRESOURCES
        }
    }
}


def parse_args():
    parser = argparse.ArgumentParser(description='Tool for test generation RS configs.')
    parser.add_argument('--generation', type=int, required=True)
    parser.add_argument('--tier', type=str, required=True)
    parser.add_argument('--location', type=str, required=True)
    parser.add_argument('--topology', type=str, default='stable-145-r1114')
    parser.add_argument('--namespace-prefix', type=str, default='/web/prod/yt')
    parser.add_argument('--chunk-namespace-prefix', type=str, default='/web/prod/chunks')
    parser.add_argument('--output-path', type=str, default='generated')

    options = parser.parse_args()
    if os.path.exists(options.output_path) and os.listdir(options.output_path):
        logging.error('Dir `{}` in --output-path is not empty'.format(options.output_path))
        sys.exit(1)

    return options


def main():
    configure_logging(debug=True)
    options = parse_args()

    gencfg_api.use_local_cache()

    tier_location_env = TIER_LOCATIONS_ENV[options.tier][options.location]
    configs = lib.generate_configs(
        generation=options.generation,
        location=options.location,
        topology=options.topology,
        remote_storage_group=tier_location_env['remote_storage_group'],
        basesearch_groups=tier_location_env['basesearch_groups'],
        basesearch_hamster_groups=tier_location_env['basesearch_hamster_groups'],
        namespace_prefix=options.namespace_prefix,
        chunk_namespace_prefix=options.chunk_namespace_prefix,
        subresources=tier_location_env['subresources'],
        tier_name=options.tier,
        endpoint_set_template=tier_location_env['endpoint_set_template'],
        endpoint_set_hamster_template=tier_location_env['endpoint_set_hamster_template'],
    )

    # Dump all configs.
    for config_name, config in configs.iteritems():
        if not os.path.exists(options.output_path):
            os.mkdir(options.output_path)
        with open(os.path.join(options.output_path, config_name), 'w') as f:
            f.write(config)

    # Dump one sample config.
    # _, config = configs.iteritems().next()
    # with open('sample.proto', 'w') as f:
    #     f.write(config)

    # Generate DNS Cache.
    # dns_cache = lib.generate_dns_cache(
    #     topology='stable-143-r17',
    #     remote_storage_group='VLA_WEB_TIER1_REMOTE_STORAGE_BASE_PIP',
    #     basesearch_groups=['VLA_WEB_TIER1_JUPITER_BASE_PIP'],
    # )
    # import google.protobuf.text_format as text_format
    # import os
    # general_dns_cache_path = os.path.join('.', 'general_dns_cache.proto')
    # with open(general_dns_cache_path, 'wb') as f:
    #     f.write(text_format.MessageToString(dns_cache))


def configure_logging(debug):
    level = logging.DEBUG if debug else logging.INFO
    formatter = logging.Formatter('%(asctime)s [%(levelname).1s] [%(name)s] %(message)s')
    console_handler = logging.StreamHandler()
    console_handler.setLevel(level)
    console_handler.setFormatter(formatter)
    logging.getLogger().addHandler(console_handler)
    logging.getLogger().setLevel(level)
