import logging
import argparse

import infra.callisto.deploy.resource as deploy_resource
import infra.callisto.deploy.tracker.client as client
import infra.callisto.controllers.sdk.tier as tiers
import infra.callisto.controllers.sdk.resource as sdk_resource


HELP_MESSAGE = [
    'This is cajuper tracker client. It resolves resource by its namespace and name',
    'If you know exact namespace, pass it with --namespace argument',
    'If you resolve shard and you know namespace prefix (such as web/prod, video/prod) pass it as first argument',
    'If you resolve shard and you know only its name, the namespace would be guessed',
    'Common usage:',
    '1. tracker_client --namespace web/prod/1552865784/WebTier1 primus-WebTier1-0-252-1552865784',
    '2. tracker_client web/prod primus-WebTier1-0-252-1552865784',
    '3. tracker_client primus-WebTier1-0-252-1552865784',
    '',
    'Extra options:',
    '1. --download-recursive - special way to download resource: fetch all subdirectories of resource'
    ' and download them one by one',
]
HELP_MESSAGE = '\n'.join(HELP_MESSAGE)


def parse_args():
    parser = argparse.ArgumentParser(description=HELP_MESSAGE, formatter_class=argparse.RawTextHelpFormatter)
    parser.add_argument('namespace_prefix', help='namespace prefix e.g. /web/prod/ or /web/betas/rthub1/', nargs='?')
    parser.add_argument('resource_name', help='resource name')
    parser.add_argument('--tracker-url', help='tracker url', default='http://vla-tracker-cajuper.n.yandex-team.ru')
    parser.add_argument('--namespace', help='exact resource namespace (e.g. /web/prod/12345678/PlatinumTier0/)')
    parser.add_argument('--download-recursive', help='get resource by downloading its subresources', action='store_true')
    parser.add_argument('--download-path', help='path to download resource', default='./')

    return parser.parse_args()


def _guess_namespace_prefix(shard):
    if shard.tier.name in [
        'WebTier1', 'PlatinumTier0', 'MsuseardataJupiterTier0',
        'EmbeddingWebTier1', 'InvertedIndexWebTier1', 'GeminiTier', 'JudTier',
    ]:
        return '/web/prod/'
    if shard.tier.name in ['WebFreshTier']:
        return '/web/callisto/'
    if shard.tier.name in ['ImgTier0', 'ImgTier1', 'ImgMmetaTier0']:
        return '/images/prod/'
    if shard.tier.name in ['ImgRIMTier']:
        return '/images/prod/rim/'
    if shard.tier.name in ['VideoTier0', 'VidMmetaTier0', 'VideoPlatinum']:
        return '/video/prod/'
    raise ValueError('cannot guess namespace-prefix')


def _guess_namespace(namespace_prefix, resource_name):
    resource_name = resource_name.split('/')[0]
    try:
        shard = tiers.parse_shard(resource_name)
    except ValueError:
        logging.error('could not guess namespace from resource-name and namespace-prefix, use --namespace')
        raise
    if namespace_prefix is None:
        logging.warning('trying to guess namespace-prefix by shard name')
        namespace_prefix = _guess_namespace_prefix(shard)
    return sdk_resource.shard_to_resource(namespace_prefix, shard).namespace


def _get_namespace_and_name(resource_name, namespace_prefix, namespace):
    name = deploy_resource.normalize_resource_name(resource_name)
    if namespace is None:
        namespace = _guess_namespace(namespace_prefix, name)
    else:
        namespace = deploy_resource.normalize_namespace(namespace)
    return namespace, name


def _resolve_mode(args):
    try:
        namespace, name = _get_namespace_and_name(args.resource_name, args.namespace_prefix, args.namespace)
        resolved_resource = client.Client(args.tracker_url).resolve_one(namespace, name)
        print resolved_resource.rbtorrent
    except Exception as exc:
        logging.error(exc.message)
        exit(1)


def _download_recursive_mode(args):
    try:
        namespace, name = _get_namespace_and_name(args.resource_name, args.namespace_prefix, args.namespace)
        resolved_resources = client.Client(args.tracker_url).match_resources(namespace, name)
        assert resolved_resources, 'no resources found in tracker'
        logging.warning('matched %s resources', len(resolved_resources))
        client.download_recursive(args.download_path, resolved_resources)
    except Exception as exc:
        logging.error(exc.message)
        exit(1)


def main():
    args = parse_args()

    if args.download_recursive:
        _download_recursive_mode(args)
    else:
        _resolve_mode(args)


if __name__ == '__main__':
    main()
