import argparse
import collections
import grequests
import json
import logging
import socket
import urlparse

from BaseHTTPServer import BaseHTTPRequestHandler, HTTPServer


def configure_logging():
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--port', help='Port', type=int, required=True)
    parser.add_argument('--config', help='Config', required=True)
    return parser.parse_args()


Instance = collections.namedtuple('Instance', ['port', 'tags'])


class ConfigParser(object):
    def __init__(self, config_filename):
        self.config_filename = config_filename

        self._open_config()
        self._update_data()

    def _open_config(self):
        self.config_fp = open(self.config_filename)

    def _update_data(self):
        self.instances = [
            Instance(
                port=workload['spec'].get('monitoring_port', workload['spec']['port']),
                tags=workload['meta'].get('tags', {})
            )
            for workload in json.load(self.config_fp)['workloads']
        ]


class HTTPServer6(HTTPServer):
    address_family = socket.AF_INET6


class ReqHandle(BaseHTTPRequestHandler):
    def _set_headers(self):
        self.send_response(200)
        self.send_header('Content-type', 'application/json')
        self.end_headers()

    def do_GET(self):
        self._set_headers()

        if self.path.startswith('/admin'):
            url = urlparse.urlparse(self.path)
            query = urlparse.parse_qsl(url.query)
            if url.path == '/admin' and ('action', 'shutdown') in query:
                self._shutdown()
            responses = {}
        else:
            responses = self._make_subrequests()

        self.wfile.write(json.dumps(responses))

    def _shutdown(self):
        import threading

        killer = threading.Thread(target=self.server.shutdown)
        killer.setDaemon(True)
        killer.start()

    def _make_subrequests(self):
        urls = [
            'http://localhost:{PORT}{PATH}'.format(PORT=instance.port, PATH=self.path)
            for instance in self.server.instancer.instances
        ]

        responses = grequests.map(
            grequests.get(url, timeout=0.5) for url in urls
        )

        values = {
            urlparse.urlparse(r.request.url).port: r.json()
            for r in responses if r
        }

        result = []

        for i in self.server.instancer.instances:
            if i.port in values:
                result.extend(self._add_tags(values[i.port], i.tags))

        return result

    def _add_tags(self, signals, tags):
        tags_string = ';'.join(
            [
                '{}={}'.format(tag_name, tag_val[-1] if isinstance(tag_val, list) else tag_val)
                for tag_name, tag_val in tags.iteritems()
            ]
        )

        return [
            [tags_string + ';' + sig[0], sig[1]]
            for sig in signals
        ]


def run(instancer, server_class=HTTPServer6, handler_class=ReqHandle, port=80):
    server_address = ('', port)
    server_class.instancer = instancer
    httpd = server_class(server_address, handler_class)
    logging.debug('Starting httpd...')
    httpd.serve_forever()


def main():
    configure_logging()
    args = parse_args()

    instances_cfg = ConfigParser(args.config)

    run(port=args.port, instancer=instances_cfg)
