import argparse
import json
import logging
import sys

import infra.callisto.libraries.yt as yt_utils


def configure_logging():
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--yt-cluster', required=True)
    parser.add_argument('--object-table', required=True)
    parser.add_argument('--nanny-service', required=True)
    parser.add_argument('--deploy-service', required=True)

    return parser.parse_args()


def _update_status(nanny_service, object_table):
    local_status = _read_local_status()
    logging.info('Local status:\n%s', local_status)

    head = object_table.head(nanny_service)
    last_status = json.loads(head['status']) if head else None
    if local_status != last_status:
        logging.info('Service status changed:\n%s\n%s', last_status, local_status)
        object_table.write(nanny_service, json.dumps(local_status))
    else:
        logging.info('Status of service not changed')
    return local_status


def _synchronized(local_status, object_table, deploy_service):
    head = object_table.head(deploy_service)
    remote_state = json.loads(head['status']) if head else {}

    synchronized_resources_ids = remote_state['RESOURCES_SYNCHRONIZE']
    remote_status = remote_state['RESOURCES']

    logging.info('Remote status:\n%s', remote_status)

    for resource in local_status:
        if resource not in synchronized_resources_ids:
            logging.info('Skip non-synchronized %s', resource)
            continue
        if resource not in remote_status:
            logging.info('Resource %s not found in remote status', resource)
            return False
        if 'skynet_id' not in local_status[resource]:
            logging.info('skynet_id of resource %s not found in local status', resource)
            return False
        if 'skynet_id' not in remote_status[resource]:
            logging.info('skynet_id of resource %s not found in remote status', resource)
            return False
        if local_status[resource]['skynet_id'] != remote_status[resource]['skynet_id']:
            logging.info('skynet_id of resource %s differs in local and remote status', resource)
            return False
    logging.info('Local and remote statuses synchronized')
    return True


def _read_local_status():
    status = {}
    for name, url in _read_dump_json().items():
        status[name] = {'skynet_id': url}
    return status


def _read_dump_json():
    with open('./dump.json') as f:
        return {name: value['urls'][0] for name, value in json.load(f)['resources'].items()}


def main():
    configure_logging()
    args = parse_args()

    status_table = yt_utils.ObjectStatusTable(yt_utils.create_yt_client(args.yt_cluster, use_rpc=True),
                                              args.object_table, readonly=False)
    status = _update_status(args.nanny_service, status_table)
    sys.exit(0 if _synchronized(status, status_table, args.deploy_service) else 1)
