# Ad-hoc intlookup parser.
# Because you cannot build yp client and gencfg core into a single binary.

import infra.callisto.controllers.sdk.tier as tiers

import collections
import json
import logging


Intlookup = collections.namedtuple(
    'Intlookup',
    ['tier', 'shard_hosts', 'instance_count', 'ints', 'ints_l2']
)


def parse(filename):
    shard_hosts = collections.defaultdict(list)
    parse.instance_count = 0

    def instance(host, _port, shard_id, _replica):
        parse.instance_count += 1
        shard_hosts[shard_id].append(host)

    tier, (ints, ints_l2) = _parse(filename, instance)
    return Intlookup(tiers.get_tier_by_name(tier) if tier else None, shard_hosts, parse.instance_count, ints, ints_l2)


def _parse(filename, on_instance):
    with open(filename) as f:
        intlookup = json.load(f)
        logging.info('load intlookup %s', intlookup[0]['file_name'])  # test if file at least looks like intlookup

        head = intlookup[0]
        hosts_per_group = int(head.get('hosts_per_group', 1))
        tier = head['tiers'][0] if head['tiers'] else None
        return tier, _read_intl2_groups(intlookup[1:], hosts_per_group, on_instance)


def _read_intl2_groups(data, hosts_per_group, on_instance):
    sequential_multishard_id = -1
    ints = {}
    ints_l2 = {}
    for intl2group_id, intl2group in enumerate(data):
        # intl2s = intl2group[0]
        ints_l2[len(ints_l2)] = _get_ints_hosts(intl2group[0])
        for multishard_id, multishard in enumerate(intl2group[1:]):
            sequential_multishard_id += 1
            int_shard_id = len(ints)
            ints[int_shard_id] = []
            for intgroup_id, intgroup in enumerate(multishard):
                # intgroup[0]: {"power":760.0}
                # ints = intgroup[1]
                int_hosts = _get_ints_hosts(intgroup[1])
                ints[int_shard_id].extend(int_hosts)
                for base_id, base in enumerate(intgroup[2:]):
                    shard_id = sequential_multishard_id * hosts_per_group + base_id
                    for line in base:
                        host, port = line.split(':')[:2]
                        on_instance(host, port, shard_id, intgroup_id)
    return ints, ints_l2


def _get_ints_hosts(ints):
    return [i.split(':')[0] for i in ints]
