
import copy
import logging
import requests
from collections import OrderedDict, defaultdict

import urllib3
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

logger = logging.getLogger('infra.dispenser_requests')


class DispenserApi(object):

    def __init__(self, dispenser_token=None, session_id=None, timeout=None):

        self.api_url = "https://dispenser.yandex-team.ru/common/api/v1"
        self.timeout = timeout
        self.session = requests.Session()
        self.session.verify = False
        if dispenser_token:
            logger.info('dispenser token authorization')
            self.session.headers['Authorization'] = 'OAuth {}'.format(dispenser_token)
        elif session_id:
            logger.info('dispenser session id authorization')
            self.session.cookies['Session_id'] = session_id
        else:
            logger.info('dispenser no authorization')

    def get_request(self, path, **kwargs):
        url = self.api_url + path
        return self.session.get(url, timeout=self.timeout, **kwargs) if self.timeout else self.session.get(url, **kwargs)

    @staticmethod
    def get_dc_segment(segments):
        segment = 'default'
        dc = 'unknown'

        dc_list = ['man', 'sas', 'vla', 'myt', 'sas']
        yt_clusters = {
            'hahn': ('hahn', 'sas'),
            'arnold': ('arnold', 'vla'),
            'seneca-vla': ('seneca', 'vla'),
            'seneca-sas': ('seneca', 'sas'),
            'seneca-man': ('seneca', 'man'),
            'bohr': ('bohr', 'sas'),
            'landau': ('landau', 'vla'),
            'vanga': ('vanga', 'unknown'),
            'markov': ('markov', 'unknown'),
            'zeno': ('zeno', 'man'),
        }

        if isinstance(segments, list):
            for row in segments:
                row_lower = row.lower()
                if row_lower in yt_clusters:
                    segment, dc = yt_clusters[row_lower]
                    break
                if row_lower in dc_list:
                    dc = row_lower
                    continue
                segment = row_lower
        else:
            segment = segments

        return segment, dc

    def get_limits(self, campaigns):
        campaign_ids = list(map(int, campaigns))

        data = []
        next_from = 0
        while next_from is not None:
            response = self.get_request('/base-resource-limits?from={}'.format(next_from))
            response.raise_for_status()
            response_data = response.json()
            next_from = response_data.get('nextFrom')
            data.extend(response_data['limits'])

        return [f for f in data if f['campaignId'] in campaign_ids]

    def get_user_res_prefix_provider(self, provider, res_name, dc):
        local_mappings = {
            ('saas.compute', ): ('yp.compute', 'u_saas_'),
            ('gencfg.compute',): ('yp.compute', 'u_gencfg_'),
            ('logbroker_yt_proxy.compute', 'data_flow_binary', 'sas'): ('yt.hahn.compute', 'u_'),
            ('logbroker_yt_proxy.compute', 'data_flow_binary', 'vla'): ('yt.arnold.compute', 'u_'),
            ('nirvana.gpu', 'gpu', 'sas'): ('yt.hahn.gpu', 'u_nirvana_'),
            ('nirvana.gpu', 'gpu', 'vla'): ('yt.arnold.gpu', 'u_nirvana_'),
            ('nirvana.compute', 'cpu'): ('yt.hahn.compute', 'u_nirvana_'),
            ('nirvana.compute', 's3-storage'): ('mds.compute', 'u_nirvana_'),
        }

        if (provider, res_name, dc) in local_mappings:
            return local_mappings[(provider, res_name, dc)]
        elif (provider, res_name) in local_mappings:
            return local_mappings[(provider, res_name)]
        elif (provider, ) in local_mappings:
            return local_mappings[(provider, )]
        else:
            return provider, 'u_'

    def filter_quota_requests(self, quota_data, date_filters):
        ans = []
        date_filters_tuple = tuple(date_filters)
        for line in quota_data:
            new_line = copy.deepcopy(line)
            new_line['resources'] = [f for f in new_line.get('resources') or [] if f['big_order'].endswith(date_filters_tuple)]
            new_line['user_resources'] = [f for f in new_line.get('user_resources') or [] if f['big_order'].endswith(date_filters_tuple)]
            if new_line['resources'] or new_line['user_resources']:
                ans.append(new_line)
        return ans

    def multiply_limits(self, quota_limits, multiplier):
        ans = []
        for line in quota_limits:
            new_line = copy.deepcopy(line)
            new_line['limit']['value'] = float(new_line['limit']['value']) * multiplier
            ans.append(new_line)
        return ans

    def fill_abc_service_parents(self, abc_tree, service_id):
        service_sequence = [service_id] + abc_tree[service_id]['parents']
        service_sequence = list(reversed(service_sequence))
        answer = []
        for i in range(10):
            key = 'ABC_SERVICE_{}'.format(i + 1)
            if i < len(service_sequence):
                local_service_id = service_sequence[i]
                name = abc_tree[local_service_id]['name_en']
            else:
                name = ''
            answer.append((key, name))
        return answer

    def get_quota_requests(self, campaigns, filters=None, abc_tree=None):

        result = []

        string_params = '?' + '&'.join(['campaign={}'.format(f) for f in campaigns])
        string_params += '&status=NEW&status=CONFIRMED&status=READY_FOR_REVIEW&status=COMPLETED&status=NEED_INFO&status=APPROVED'
        string_params += '&expand=BASE_RESOURCES&expand=EXTRA_REPORT_FIELDS'
        response = self.get_request("/quota-requests" + string_params)
        response.raise_for_status()
        data = response.json()['result']

        for row in data:
            extra_fields = row['extraReportFields']
            abc_service_id = row['project']['abcServiceId']
            detail_array = [
                ('SUBTICKET', row['trackerIssueKey']),
                ('SUBTICKET_URL', 'https://st.yandex-team.ru/{}'.format(row['trackerIssueKey'])),
                ('ID', row['id']),
                ('ABC_SERVICE', row['project']['name']),
                ('parent_project', row['project']['parentProjectKey']),
                ('CAMPAIGN', row['campaign']['key']),
                ('GOAL_NAME', row['goal']['name'] if row['goal'] else ''),
                ('GOAL_URL', 'https://goals.yandex-team.ru/filter?goal={}'.format(row['goal']['id']) if row['goal'] else ''),
                ('AUTHOR', row['author']),
                ('RESPONSIBLES', row['responsible']),
                ('HEAD_1', ','.join(extra_fields.get('headFirst', []))),
                ('HEAD_DEPARTMENT_1', extra_fields['headDepartmentFirst']['name'] if extra_fields['headDepartmentFirst'] else ''),
                ('HEAD_2', ','.join(extra_fields.get('headSecond', []))),
                ('HEAD_DEPARTMENT_2', extra_fields['headDepartmentSecond']['name'] if extra_fields['headDepartmentSecond'] else ''),
                ('HEAD_3', ','.join(extra_fields.get('headThird', []))),
                ('HEAD_DEPARTMENT_3', extra_fields['headDepartmentThird']['name'] if extra_fields['headDepartmentThird'] else ''),
                ('JUSTIFICATION', row['resourcePreorderReasonType']),
                ('order_type', row['type']),
                ('STATUS', row['status']),
                ('SUMMARY', row['summary']),
                ('Value Stream', extra_fields['valueStream']['name'] if extra_fields['valueStream'] else ''),
                ('VS Manager', ','.join(extra_fields.get('valueStreamManager', []))),
                ('VS Leader', ','.join(extra_fields.get('valueStreamLeader', []))),
                ('VS Capacity Planner', ','.join(extra_fields.get('valueStreamCapacityPlanner', []))),
                ('BOT_PREORDER', ','.join(['{}'.format(f) for f in row['botPreOrderIds']])),
                ('PROVIDER', ''),
                ('DC', ''),
            ]
            detail_array.extend(self.fill_abc_service_parents(abc_tree, abc_service_id))
            details = OrderedDict(detail_array)

            if filters:
                skip = False
                for filter_key, filter_vals in filters.items():
                    if filter_key in details and details[filter_key] not in filter_vals:
                        skip = True
                        break
                if skip:
                    continue

            provider_dc_res = defaultdict(lambda: defaultdict(list))
            for res in row.get('baseResourceChanges', {}):
                provider = res['baseResource']['type']['provider']['key']
                segment, dc = self.get_dc_segment(res['baseResource'].get('segmentKeys', []))

                res_type = 'gpu' if 'gpu' in res['baseResource']['key'] else 'compute'
                if provider == 'yt':
                    provider = '{}.{}.{}'.format(provider, segment, res_type)
                else:
                    provider = '{}.{}'.format(provider, res_type)

                provider_dc_res[(provider, dc)]['base_res'].append(
                    {
                        'segment': segment,
                        'resource': res['baseResource']['key'],
                        'big_order': res['bigOrder']['date'],
                        'res_id': res['baseResource']['id'],
                        'unit': res['totalAmount']['unit'],
                        'val': res['totalAmount']['value'],
                    }
                )

            for res in row.get('changes', {}):
                provider = res['service']['key']
                res_name = res['resource']['key']

                segment, dc = self.get_dc_segment(res.get('segmentKeys', []))

                res_type = 'gpu' if 'gpu' in res['resource']['key'] else 'compute'
                if provider == 'yt':
                    provider = '{}.{}.{}'.format(provider, segment, res_type)
                else:
                    provider = '{}.{}'.format(provider, res_type)

                provider, prefix = self.get_user_res_prefix_provider(provider, res_name, dc)

                provider_dc_res[(provider, dc)]['user_res'].append(
                    {
                        'segment': segment,
                        'resource': '{}{}'.format(prefix, res_name),
                        'big_order': res['order']['orderDate'],
                        'unit': res['amount']['unit'],
                        'val': res['amount']['value'],
                        'cost': int(res['owningCost']) if res['owningCost'] else 0,
                    }
                )

            for provider_dc, res in provider_dc_res.items():
                provider, dc = provider_dc
                result_details = copy.deepcopy(details)
                result_details['PROVIDER'] = provider
                result_details['DC'] = dc
                result.append({
                    'details': result_details,
                    'resources': res.get('base_res'),
                    'user_resources': res.get('user_res'),
                })

        return result
