# -*- coding: utf-8 -*-

import os
import copy
import shutil
import tempfile
import openpyxl
import pandas as pd
from collections import defaultdict, OrderedDict

from .common import (
    is_close,
    excel_colnum_string,
    remove_prefix,
    convert_values,
    prepare_simple_weights
)


class TableData(object):

    def __init__(self, data, limits, targets, base_costs=None, yoy_capex=None):

        self.initial_data = data
        self.cost_targets = targets
        self.base_costs = base_costs if base_costs is not None else {}
        self.yoy_capex = float(yoy_capex) if yoy_capex else 0.0
        self.detail_columns, self.detail_columns_set = self._detail_columns()
        self.compact_totals_columns = []
        self.compact_totals_rows = []
        self.totals, self.res_id_mappings, self.dc_res_id_mappings = self._calculate_totals()
        self.limits = self._process_limits(limits)
        (
            self.per_provider_columns,
            self.per_provider_rows,
            self.print_totals,
            self.print_per_dc_totals,
            self.print_units
        ) = self.per_provider_data()
        self.simple_weights, self.u_base_res_mappings = prepare_simple_weights(self.print_totals)
        self.delta_correction_coeffs, self.target_correction_coeffs = self.calculate_delta_correction()
        self.per_provider_sort_res = self.per_provider_add_user_res()
        (
            self.print_columns,
            self.print_rows,
            self.per_provider_print_columns,
            self.per_provider_print_rows,
            self.per_provider_res_columns
        ) = self.prepare_excel_data()
        (
            self.costs_columns,
            self.costs_rows,
            self.monthly_costs_columns,
            self.monthly_costs_rows,
            self.based_monthly_costs_columns,
            self.based_monthly_costs_rows,
            self.yoy_monthly_costs_columns,
            self.yoy_monthly_costs_rows,
            self.costs_sort_order
        ) = self._calculate_cost_table()
        self.monthly_cost_pivot = self._prepare_monthly_cost_pivot()
        self.monthly_delta_cost_pivot = self._prepare_monthly_cost_pivot(prefix='delta_')
        self.monthly_target_cost_pivot = self._prepare_monthly_cost_pivot(prefix='target_')
        self.provider_monthly_cost_pivot = self._prepare_monthly_cost_pivot(mid_name='provider_')
        self.provider_monthly_delta_cost_pivot = self._prepare_monthly_cost_pivot(prefix='delta_', mid_name='provider_')
        self.provider_monthly_target_cost_pivot = self._prepare_monthly_cost_pivot(prefix='target_',
                                                                                   mid_name='provider_')
        self.network_monthly_cost_pivot = self._prepare_monthly_cost_pivot(mid_name='network_')
        self.network_monthly_delta_cost_pivot = self._prepare_monthly_cost_pivot(prefix='delta_', mid_name='network_')
        self.provider_cost_pivot = self._prepare_cost_pivot(filter_mode='provider')
        self.provider_gpu_cost_pivot = self._prepare_cost_pivot(filter_mode='provider_gpu')
        self.provider_no_gpu_cost_pivot = self._prepare_cost_pivot(filter_mode='provider_no_gpu')
        self.network_cost_pivot = self._prepare_cost_pivot(filter_mode='network')
        self.cost_pivot = self._prepare_cost_pivot()
        self.provider_res_weight_tops = self.prepare_compact_total_rows()
        self.pivot_data, self.based_pivot_data, self.yoy_pivot_data = self.prepare_pivot_data()

    def _detail_columns(self):
        detail_columns = list(self.initial_data[0]['details'].keys()) if self.initial_data else []
        detail_columns_set = set(detail_columns)
        return detail_columns, detail_columns_set

    @staticmethod
    def _process_limits(limits):
        return {(f['baseResourceId'], f['limit']['unit']): f['limit']['value'] for f in limits}

    def calculate_target_cost(self, cost, delta_cost, bu, vs):
        if not is_close(delta_cost, 0.0):
            if self.delta_correction_coeffs[bu][vs] < 1.0:
                delta_cost *= self.delta_correction_coeffs[bu][vs]
                target_cost = cost - delta_cost
            else:
                target_cost = cost - delta_cost
                target_cost *= self.target_correction_coeffs[bu][vs]
                delta_cost = cost - target_cost
        else:
            target_cost = cost - delta_cost
        return target_cost, delta_cost

    def _calculate_totals(self):
        totals = defaultdict(float)
        provider_res_mapping = defaultdict(set)
        provider_dc_res_mapping = defaultdict(set)

        for row in self.initial_data:
            provider = row['details']['PROVIDER']
            dc = row['details']['DC']
            for res_row in row.get('resources') or []:
                res_name = res_row['resource'].split(':')[1]
                totals[(res_row['res_id'], res_row['unit'])] += res_row['val']
                provider_res_mapping[(provider, res_name)].add((res_row['res_id']))
                provider_dc_res_mapping[(provider, res_name, dc)].add((res_row['res_id']))
        res_id_mappings = {res_id: mapping for mapping in provider_res_mapping.values() for res_id in mapping}
        dc_res_id_mappings = {res_id: mapping for mapping in provider_dc_res_mapping.values() for res_id in mapping}

        return totals, res_id_mappings, dc_res_id_mappings

    def get_delta_cost(self, provider, res_name, cost):
        key = 'yt.' if provider.startswith('yt.') else provider
        simple_res_name = self.u_base_res_mappings.get(key, {}).get(res_name, remove_prefix(res_name, 'u_'))
        coeff = self.simple_weights[provider].get(simple_res_name, 0.0)
        return (cost * (coeff - 1.0) / coeff) if coeff > 1.0 else 0.0

    def add_user_details_to_excel_row(self, row, per_provider_user_res_columns, per_provider_sort_res,
                                      per_provider_columns, resources, provider):
        def process_row(column_name, val):
            per_provider_columns[provider].add(column_name)
            per_provider_user_res_columns[provider].add(column_name)
            if column_name in row:
                row[column_name] += val
            else:
                row[column_name] = val

        res_name = resources['resource']
        val = resources['val']
        bu = row['ABC_SERVICE_1']
        vs = row['ABC_SERVICE_2']
        unit = resources['unit']
        conv_val, conv_unit = convert_values(val, unit)

        cost = resources['cost']
        delta_cost = self.get_delta_cost(provider, res_name, cost)
        target_cost, delta_cost = self.calculate_target_cost(cost, delta_cost, bu, vs)

        # raw values
        process_row('{}_{} ({})'.format(provider, res_name, conv_unit), conv_val)
        process_row('{}_{} (₽/m)'.format(provider, res_name), cost)
        process_row('{}_cost (₽/m)'.format(provider), cost)

        process_row('{}_{} cut (₽/m)'.format(provider, res_name), delta_cost)
        process_row('{}_cost cut (₽/m)'.format(provider), delta_cost)

        process_row('{}_{} target (₽/m)'.format(provider, res_name), target_cost)
        process_row('{}_cost target (₽/m)'.format(provider), target_cost)

        per_provider_sort_res[provider][(res_name, conv_unit)] += delta_cost

    def add_details_to_excel_row(self, row, print_totals, per_dc_totals, per_provider_columns, resources, provider, dc):
        def process_row(column_name, val, count_per_dc=True):
            per_provider_columns[provider].add(column_name)
            row[column_name] = val
            print_totals[provider][res_name][column_name] += val
            if count_per_dc:
                per_dc_totals[provider][res_name][dc][column_name] += val

        res_id = resources['res_id']
        res_name = resources['res_name']
        segment = resources['segment']
        val = resources['val']
        unit = resources['unit']
        conv_val, conv_unit = convert_values(val, unit)

        val_total = self.totals.get((res_id, unit), 0.0)
        val_limit = self.limits.get((res_id, unit), 0.0)
        coeff = val_limit / val_total if val_total else 0.0

        dc_res_ids = self.dc_res_id_mappings.get(res_id, [])
        dc_val_total = sum(self.totals.get((dc_res_id, unit), 0.0) for dc_res_id in dc_res_ids)
        dc_val_limit = sum(self.limits.get((dc_res_id, unit), 0.0) for dc_res_id in dc_res_ids)
        dc_coeff = dc_val_limit / dc_val_total if dc_val_total else 0.0

        res_res_ids = self.res_id_mappings.get(res_id, [])
        res_val_total = sum(self.totals.get((res_res_id, unit), 0.0) for res_res_id in res_res_ids)
        res_val_limit = sum(self.limits.get((res_res_id, unit), 0.0) for res_res_id in res_res_ids)
        res_coeff = res_val_limit / res_val_total if res_val_total else 0.0

        # raw values
        process_row('{}_{}_{}_{} ({})'.format(provider, res_name, dc, segment, conv_unit), conv_val)
        process_row('{}_{}_{}_{}_limit ({})'.format(provider, res_name, dc, segment, conv_unit), conv_val * coeff)
        process_row('{}_{}_{}_{} (w)'.format(provider, res_name, dc, segment), val / val_limit if val_limit else 0.0)

        # provider dc totals
        process_row('{}_{}_{} ({})'.format(provider, res_name, dc, conv_unit), conv_val)
        process_row('{}_{}_{}_limit ({})'.format(provider, res_name, dc, conv_unit), conv_val * dc_coeff)
        process_row('{}_{}_{} (w)'.format(provider, res_name, dc), val / dc_val_limit if dc_val_limit else 0.0)

        # provider total
        process_row('{}_{} ({})'.format(provider, res_name, conv_unit), conv_val, count_per_dc=False)
        process_row('{}_{}_limit ({})'.format(provider, res_name, conv_unit), conv_val * res_coeff, count_per_dc=False)
        process_row('{}_{} (w)'.format(provider, res_name), val / res_val_limit if res_val_limit else 0.0,
                    count_per_dc=False)

    def get_excel_cell(self, row, column, detail_fields):
        return row.get(column, '' if column in detail_fields else 0.0)

    def compact_resources(self, row):
        compact_resource_rows = {}
        for res_row in row.get('resources') or []:
            res_name = res_row['resource'].split(':')[1]
            res_key = tuple(sorted(self.dc_res_id_mappings[res_row['res_id']]))
            if res_key in compact_resource_rows:
                compact_resource_rows[res_key]['val'] += res_row['val']
            else:
                compact_resource_rows[res_key] = copy.deepcopy(res_row)
                compact_resource_rows[res_key]['res_name'] = res_name
        return compact_resource_rows

    def compact_user_resources(self, row):
        compacted_user_rows = {}
        for user_res_row in row.get('user_resources') or []:
            compaction_key = (user_res_row['resource'], user_res_row['segment'])
            if compaction_key in compacted_user_rows:
                compacted_user_rows[compaction_key]['val'] += user_res_row['val']
                compacted_user_rows[compaction_key]['cost'] += user_res_row['cost']
            else:
                compacted_user_rows[compaction_key] = copy.deepcopy(user_res_row)
        return compacted_user_rows

    def per_provider_data(self):
        print_totals = defaultdict(lambda: defaultdict(lambda: defaultdict(float)))
        print_per_dc_totals = defaultdict(lambda: defaultdict(lambda: defaultdict(lambda: defaultdict(float))))
        per_provider_columns = defaultdict(set)
        per_provider_rows = defaultdict(list)
        print_units = {}

        for row in self.initial_data:
            dc = row['details']['DC']
            provider = row['details']['PROVIDER']
            new_row = copy.deepcopy(row['details'])
            for resources in self.compact_resources(row).values():
                res_name = resources['res_name']
                unit = resources['unit']
                self.add_details_to_excel_row(new_row, print_totals, print_per_dc_totals, per_provider_columns,
                                              resources, provider, dc)
                if (provider, res_name) not in print_units:
                    print_units[(provider, res_name)] = convert_values(0.0, unit)[1]
            per_provider_rows[provider].append(new_row)
        return per_provider_columns, per_provider_rows, print_totals, print_per_dc_totals, print_units

    def per_provider_add_user_res(self):
        per_provider_user_res_columns = defaultdict(set)
        per_provider_sort_res = defaultdict(lambda: defaultdict(float))
        for row in self.initial_data:
            provider = row['details']['PROVIDER']
            new_row = copy.deepcopy(row['details'])
            for resources in self.compact_user_resources(row).values():
                res_name = resources['resource']
                unit = resources['unit']
                self.add_user_details_to_excel_row(new_row, per_provider_user_res_columns, per_provider_sort_res,
                                                   self.per_provider_columns, resources, provider)
                if (provider, res_name) not in self.print_units:
                    self.print_units[(provider, res_name)] = convert_values(0.0, unit)[1]
            self.per_provider_rows[provider].append(new_row)
        return per_provider_sort_res

    def prepare_excel_data(self):
        print_columns = (
            self.detail_columns
            + [col_name for _, provider_data in self.per_provider_columns.items() for col_name in sorted(provider_data)]
        )
        print_rows = []
        for provider_rows in self.per_provider_rows.values():
            for row in provider_rows:
                new_row = [self.get_excel_cell(row, column, self.detail_columns_set) for column in print_columns]
                print_rows.append(new_row)

        per_provider_print_columns = defaultdict(list)
        for provider, provider_data in self.per_provider_columns.items():
            per_provider_print_columns[provider] = self.detail_columns + list(sorted(provider_data))

        per_provider_print_rows = defaultdict(list)
        for provider, rows in self.per_provider_rows.items():
            for row in rows:
                new_row = [self.get_excel_cell(row, column, self.detail_columns_set) for column in per_provider_print_columns[provider]]
                per_provider_print_rows[provider].append(new_row)

        per_provider_res_columns = {}
        for provider, provider_data in self.per_provider_sort_res.items():
            per_provider_res_columns[provider] = [
                '{}_cost (₽/m)'.format(provider),
                '{}_cost cut (₽/m)'.format(provider),
                '{}_cost target (₽/m)'.format(provider),
            ]
            for res_data, _ in sorted(provider_data.items(), key=lambda x: x[1], reverse=True):
                res_name, conv_unit = res_data
                per_provider_res_columns[provider].append('{}_{} ({})'.format(provider, res_name, conv_unit))
                per_provider_res_columns[provider].append('{}_{} (₽/m)'.format(provider, res_name))
                per_provider_res_columns[provider].append('{}_{} cut (₽/m)'.format(provider, res_name))
                per_provider_res_columns[provider].append('{}_{} target (₽/m)'.format(provider, res_name))

        return print_columns, print_rows, per_provider_print_columns, per_provider_print_rows, per_provider_res_columns

    def add_monthly_costs(self, collection, key, mid_name, big_order, cost, delta_cost, target_cost):
        def add_cost_data(month):
            collection[key]['cost_monthly_{} (₽/m)'.format(month)] += cost
            collection[key]['cost_yearly (₽/m)'] += cost
            collection[key]['cost_{}_monthly_{} (₽/m)'.format(mid_name, month)] += cost
            collection[key]['cost_{}_yearly (₽/m)'.format(mid_name)] += cost

            collection[key]['delta_cost_monthly_{} (₽/m)'.format(month)] += delta_cost
            collection[key]['delta_cost_yearly (₽/m)'] += delta_cost
            collection[key]['delta_cost_{}_monthly_{} (₽/m)'.format(mid_name, month)] += delta_cost
            collection[key]['delta_cost_{}_yearly (₽/m)'.format(mid_name)] += delta_cost

            collection[key]['target_cost_monthly_{} (₽/m)'.format(month)] += target_cost
            collection[key]['target_cost_yearly (₽/m)'] += target_cost
            collection[key]['target_cost_{}_monthly_{} (₽/m)'.format(mid_name, month)] += target_cost
            collection[key]['target_cost_{}_yearly (₽/m)'.format(mid_name)] += target_cost

        if big_order.endswith(('-06-30', )):
            add_cost_data('06')
            add_cost_data('07')
            add_cost_data('08')
        if big_order.endswith(('-06-30', '-09-30')):
            add_cost_data('09')
            add_cost_data('10')
            add_cost_data('11')
        if big_order.endswith(('-06-30', '-09-30', '-12-31')):
            add_cost_data('12')

    @staticmethod
    def monthly_columns(prefix='', mid_name=''):
        return [
            '{}cost_{}yearly (₽/m)'.format(prefix, mid_name),
            '{}cost_{}monthly_01 (₽/m)'.format(prefix, mid_name),
            '{}cost_{}monthly_02 (₽/m)'.format(prefix, mid_name),
            '{}cost_{}monthly_03 (₽/m)'.format(prefix, mid_name),
            '{}cost_{}monthly_04 (₽/m)'.format(prefix, mid_name),
            '{}cost_{}monthly_05 (₽/m)'.format(prefix, mid_name),
            '{}cost_{}monthly_06 (₽/m)'.format(prefix, mid_name),
            '{}cost_{}monthly_07 (₽/m)'.format(prefix, mid_name),
            '{}cost_{}monthly_08 (₽/m)'.format(prefix, mid_name),
            '{}cost_{}monthly_09 (₽/m)'.format(prefix, mid_name),
            '{}cost_{}monthly_10 (₽/m)'.format(prefix, mid_name),
            '{}cost_{}monthly_11 (₽/m)'.format(prefix, mid_name),
            '{}cost_{}monthly_12 (₽/m)'.format(prefix, mid_name),
        ]

    def calculate_delta_correction(self):
        def fill_vs_coeffs(ratio, aggregator, vs_coeffs, bu_coeffs, vs_old_sums, vs_new_sums):
            for bu, bu_data in aggregator.items():
                bu_ratio = ratio
                if bu in vs_coeffs:
                    bu_ratio = vs_new_sums[bu] / vs_old_sums[bu]
                elif bu in bu_coeffs:
                    bu_ratio = bu_coeffs[bu]
                for vs in (vs for vs in bu_data if vs not in vs_coeffs[bu]):
                    vs_coeffs[bu][vs] = bu_ratio

        full_delta_aggregator = defaultdict(lambda: defaultdict(float))
        full_target_aggregator = defaultdict(lambda: defaultdict(float))
        part_delta_aggregator = defaultdict(lambda: defaultdict(float))
        part_target_aggregator = defaultdict(lambda: defaultdict(float))

        part_cost_targets = copy.deepcopy(self.cost_targets)

        for row in self.initial_data:
            provider = row['details']['PROVIDER']
            bu = row['details']['ABC_SERVICE_1']
            vs = row['details']['ABC_SERVICE_2']
            if provider == 'strm.compute':
                continue
            for user_res_row in row.get('user_resources') or []:
                res_name = user_res_row['resource']
                cost = user_res_row['cost']
                delta_cost = self.get_delta_cost(provider, res_name, cost)
                target_cost = cost - delta_cost

                full_delta_aggregator[bu][vs] += delta_cost
                full_target_aggregator[bu][vs] += target_cost
                if is_close(delta_cost, 0.0):
                    if bu in part_cost_targets:
                        part_cost_targets[bu]['value'] -= target_cost
                        if vs in part_cost_targets[bu].get('vs', {}):
                            part_cost_targets[bu]['vs'][vs]['value'] -= target_cost
                else:
                    part_delta_aggregator[bu][vs] += delta_cost
                    part_target_aggregator[bu][vs] += target_cost

        bu_delta_correction_coeffs = {}
        vs_delta_correction_coeffs = defaultdict(dict)
        old_delta_sum = sum([sum(f.values()) for f in full_delta_aggregator.values()])
        new_delta_sum = old_delta_sum
        vs_old_delta_sums = {k: sum(v.values()) for k, v in full_delta_aggregator.items()}
        vs_new_delta_sums = copy.deepcopy(vs_old_delta_sums)

        for bu, bu_data in self.cost_targets.items():
            if bu not in full_delta_aggregator:
                continue
            full_target_sum = sum(full_target_aggregator[bu].values())
            full_delta_sum = sum(full_delta_aggregator[bu].values())
            target_diff = bu_data['value'] - full_target_sum
            bu_delta_correction_coeffs[bu] = (full_delta_sum - target_diff) / full_delta_sum
            old_delta_sum -= full_delta_sum
            new_delta_sum -= (full_delta_sum - target_diff)
            vs_new_delta_sums[bu] -= target_diff
            for vs, vs_data in bu_data.get('vs', {}).items():
                if vs not in full_delta_aggregator[bu]:
                    continue
                vs_target_diff = vs_data['value'] - full_target_aggregator[bu][vs]
                vs_delta_correction_coeffs[bu][vs] = (
                    (full_delta_aggregator[bu][vs] - vs_target_diff) / full_delta_aggregator[bu][vs]
                )
                vs_old_delta_sums[bu] -= full_delta_aggregator[bu][vs]
                vs_new_delta_sums[bu] -= (full_delta_aggregator[bu][vs] - vs_target_diff)

        bu_target_correction_coeffs = {}
        vs_target_correction_coeffs = defaultdict(dict)
        old_target_sum = sum([sum(f.values()) for f in part_target_aggregator.values()])
        new_target_sum = old_target_sum
        vs_old_target_sums = {k: sum(v.values()) for k, v in part_target_aggregator.items()}
        vs_new_target_sums = copy.deepcopy(vs_old_target_sums)

        for bu, bu_data in part_cost_targets.items():
            if bu not in part_target_aggregator:
                continue
            full_val = bu_data['value']
            full_part_target_sum = sum(part_target_aggregator[bu].values())
            bu_target_correction_coeffs[bu] = full_val / full_part_target_sum
            old_target_sum -= full_part_target_sum
            new_target_sum -= full_val
            vs_new_target_sums[bu] = full_val
            for vs, vs_data in bu_data.get('vs', {}).items():
                if vs not in part_target_aggregator[bu]:
                    continue
                vs_target_correction_coeffs[bu][vs] = vs_data['value'] / part_target_aggregator[bu][vs]
                vs_old_target_sums[bu] -= part_target_aggregator[bu][vs]
                vs_new_target_sums[bu] -= vs_data['value']

        rest_delta_ratio = new_delta_sum / old_delta_sum
        fill_vs_coeffs(rest_delta_ratio, full_delta_aggregator, vs_delta_correction_coeffs,
                       bu_delta_correction_coeffs, vs_old_delta_sums, vs_new_delta_sums)

        rest_target_ratio = new_target_sum / old_target_sum
        fill_vs_coeffs(rest_target_ratio, part_target_aggregator, vs_target_correction_coeffs,
                       bu_target_correction_coeffs, vs_new_target_sums, vs_old_target_sums)

        return vs_delta_correction_coeffs, vs_target_correction_coeffs

    def add_base_costs(self, based_monthly_aggregator):
        def add_row(k, v, head1, head2, provider):
            based_monthly_aggregator[k] = defaultdict(float)
            based_monthly_aggregator[k]['SUBTICKET'] = k
            based_monthly_aggregator[k]['SUBTICKET_URL'] = k
            based_monthly_aggregator[k]['HEAD_DEPARTMENT_1'] = head1
            based_monthly_aggregator[k]['HEAD_DEPARTMENT_2'] = head2
            based_monthly_aggregator[k]['ABC_SERVICE_1'] = head1
            based_monthly_aggregator[k]['ABC_SERVICE_2'] = head2
            based_monthly_aggregator[k]['PROVIDER'] = provider
            based_monthly_aggregator[k]['order_type'] = 'BASE_COSTS'
            for month in range(1, 13):
                based_monthly_aggregator[k]['cost_monthly_{:02d} (₽/m)'.format(month)] += v
                based_monthly_aggregator[k]['cost_yearly (₽/m)'] += v
                based_monthly_aggregator[k]['target_cost_monthly_{:02d} (₽/m)'.format(month)] += v
                based_monthly_aggregator[k]['target_cost_yearly (₽/m)'] += v
                based_monthly_aggregator[k]['cost_provider_monthly_{:02d} (₽/m)'.format(month)] += v
                based_monthly_aggregator[k]['cost_provider_yearly (₽/m)'] += v
                based_monthly_aggregator[k]['target_cost_provider_monthly_{:02d} (₽/m)'.format(month)] += v
                based_monthly_aggregator[k]['target_cost_provider_yearly (₽/m)'] += v

        base_cost_id = 1
        for base_cost_type, base_costs in self.base_costs.items():
            for bu_vs, val in base_costs.items():
                bu, vs = bu_vs.split(' > ')
                key = '{}-{}'.format(base_cost_type, base_cost_id)
                add_row(key, val, bu, vs, base_cost_type)
                base_cost_id += 1

    def distribute_yoy_capex(self, yoy_monthly_aggregator, yoy_monthly_costs_columns):
        total_yoy_demand = 0.0
        total_yoy_demand_target = 0.0
        total_yoy_provider_demand = 0.0
        total_yoy_provider_demand_target = 0.0
        for row in yoy_monthly_aggregator.values():
            row['yoy_demand'] = row.get('cost_monthly_12 (₽/m)', 0.0) - row.get('cost_monthly_01 (₽/m)', 0.0)
            row['yoy_demand_target'] = row.get('target_cost_monthly_12 (₽/m)', 0.0) - row.get('cost_monthly_01 (₽/m)', 0.0)
            row['yoy_provider_demand'] = row.get('cost_provider_monthly_12 (₽/m)', 0.0) - row.get('cost_provider_monthly_01 (₽/m)', 0.0)
            row['yoy_provider_demand_target'] = row.get('target_cost_provider_monthly_12 (₽/m)', 0.0) - row.get('cost_provider_monthly_01 (₽/m)', 0.0)
            total_yoy_demand += row['yoy_demand']
            total_yoy_demand_target += row['yoy_demand_target']
            total_yoy_provider_demand += row['yoy_provider_demand']
            total_yoy_provider_demand_target += row['yoy_provider_demand_target']
        for row in yoy_monthly_aggregator.values():
            row['capex_demand'] = self.yoy_capex * row['yoy_demand'] / total_yoy_demand_target if total_yoy_demand else 0.0
            row['capex_demand_target'] = self.yoy_capex * row['yoy_demand_target'] / total_yoy_demand_target if total_yoy_demand_target else 0.0
            row['capex_provider_demand'] = self.yoy_capex * row['yoy_provider_demand'] / total_yoy_provider_demand_target if total_yoy_provider_demand else 0.0
            row['capex_provider_demand_target'] = self.yoy_capex * row['yoy_provider_demand_target'] / total_yoy_provider_demand_target if total_yoy_provider_demand_target else 0.0
        yoy_monthly_costs_columns.extend(
            [
                'yoy_demand',
                'yoy_demand_target',
                'yoy_provider_demand',
                'yoy_provider_demand_target',
                'capex_demand',
                'capex_demand_target',
                'capex_provider_demand',
                'capex_provider_demand_target',
            ]
        )

    def _calculate_cost_table(self):
        per_provider_costs_columns = defaultdict(set)
        costs_sort_order = defaultdict(lambda: defaultdict(float))
        detail_columns = []
        aggregator = {}
        monthly_detail_columns = []
        monthly_aggregator = {}
        for row in self.initial_data:
            provider = row['details']['PROVIDER']
            ticket = row['details']['SUBTICKET']
            bu = row['details']['ABC_SERVICE_1']
            vs = row['details']['ABC_SERVICE_2']

            aggregated_provider = provider
            if provider.startswith('yt.'):
                if '.compute' in provider:
                    aggregated_provider = 'yt.compute'
                elif '.gpu' in provider:
                    aggregated_provider = 'yt.gpu'

            ticket_provider_key = (ticket, aggregated_provider)
            if ticket_provider_key not in monthly_aggregator:
                new_row = defaultdict(int, row['details'])
                del new_row['DC']
                new_row['PROVIDER'] = aggregated_provider
                monthly_aggregator[ticket_provider_key] = copy.deepcopy(new_row)
                if not monthly_detail_columns:
                    monthly_detail_columns = list(row['details'].keys())
                    monthly_detail_columns.remove('DC')

            if ticket not in aggregator:
                new_row = defaultdict(int, row['details'])
                del new_row['DC']
                del new_row['PROVIDER']
                aggregator[ticket] = new_row
                if not detail_columns:
                    detail_columns = list(row['details'].keys())
                    detail_columns.remove('DC')

            for user_res_row in row.get('user_resources') or []:
                res_name = user_res_row['resource']
                suffix = 'network' if provider == 'strm.compute' else 'provider'
                if aggregated_provider in ('yp.gpu', 'yt.gpu'):
                    gpu_suffix = 'provider_gpu'
                elif provider != 'strm.compute':
                    gpu_suffix = 'provider_no_gpu'
                else:
                    gpu_suffix = 'provider_noncount'

                cost = user_res_row['cost']
                delta_cost = self.get_delta_cost(provider, res_name, cost)
                target_cost, delta_cost = self.calculate_target_cost(cost, delta_cost, bu, vs)

                per_provider_costs_columns[provider].add('cost_{}_{} (₽/m)'.format(provider, res_name))
                aggregator[ticket]['cost_{}_{} (₽/m)'.format(provider, res_name)] += cost
                aggregator[ticket]['cost_{} (₽/m)'.format(provider)] += cost
                aggregator[ticket]['cost_total (₽/m)'] += cost
                aggregator[ticket]['cost_{}_total (₽/m)'.format(suffix)] += cost
                aggregator[ticket]['cost_{}_total (₽/m)'.format(gpu_suffix)] += cost

                per_provider_costs_columns[provider].add('delta_cost_{}_{} (₽/m)'.format(provider, res_name))
                aggregator[ticket]['delta_cost_{}_{} (₽/m)'.format(provider, res_name)] += delta_cost
                aggregator[ticket]['delta_cost_{} (₽/m)'.format(provider)] += delta_cost
                aggregator[ticket]['delta_cost_total (₽/m)'] += delta_cost
                aggregator[ticket]['delta_cost_{}_total (₽/m)'.format(suffix)] += delta_cost
                aggregator[ticket]['delta_cost_{}_total (₽/m)'.format(gpu_suffix)] += delta_cost

                per_provider_costs_columns[provider].add('target_cost_{}_{} (₽/m)'.format(provider, res_name))
                aggregator[ticket]['target_cost_{}_{} (₽/m)'.format(provider, res_name)] += target_cost
                aggregator[ticket]['target_cost_{} (₽/m)'.format(provider)] += target_cost
                aggregator[ticket]['target_cost_total (₽/m)'] += target_cost
                aggregator[ticket]['target_cost_{}_total (₽/m)'.format(suffix)] += target_cost
                aggregator[ticket]['target_cost_{}_total (₽/m)'.format(gpu_suffix)] += target_cost

                self.add_monthly_costs(monthly_aggregator, (ticket, aggregated_provider), suffix,
                                       user_res_row['big_order'], cost, delta_cost, target_cost)

                aggregated_provider = provider
                if provider.startswith('yt.'):
                    if '.compute' in provider:
                        aggregated_provider = 'yt.compute'
                    elif '.gpu' in provider:
                        aggregated_provider = 'yt.gpu'
                    aggregator[ticket]['cost_{} (₽/m)'.format(aggregated_provider)] += cost
                    aggregator[ticket]['delta_cost_{} (₽/m)'.format(aggregated_provider)] += delta_cost
                    aggregator[ticket]['target_cost_{} (₽/m)'.format(aggregated_provider)] += target_cost
                    per_provider_costs_columns[aggregated_provider] = set()

                costs_sort_order[aggregated_provider][(provider, res_name)] += delta_cost

        costs_columns = copy.deepcopy(detail_columns)
        costs_columns.append('cost_total (₽/m)')
        costs_columns.append('delta_cost_total (₽/m)')
        costs_columns.append('target_cost_total (₽/m)')
        costs_columns.append('cost_network_total (₽/m)')
        costs_columns.append('delta_cost_network_total (₽/m)')
        costs_columns.append('target_cost_network_total (₽/m)')
        costs_columns.append('cost_provider_total (₽/m)')
        costs_columns.append('delta_cost_provider_total (₽/m)')
        costs_columns.append('target_cost_provider_total (₽/m)')
        costs_columns.append('cost_provider_gpu_total (₽/m)')
        costs_columns.append('delta_cost_provider_gpu_total (₽/m)')
        costs_columns.append('target_cost_provider_gpu_total (₽/m)')
        costs_columns.append('cost_provider_no_gpu_total (₽/m)')
        costs_columns.append('delta_cost_provider_no_gpu_total (₽/m)')
        costs_columns.append('target_cost_provider_no_gpu_total (₽/m)')
        for provider, col_names in per_provider_costs_columns.items():
            costs_columns.append('cost_{} (₽/m)'.format(provider))
            costs_columns.append('delta_cost_{} (₽/m)'.format(provider))
            costs_columns.append('target_cost_{} (₽/m)'.format(provider))
            for col_name in col_names:
                costs_columns.append(col_name)
        costs_rows = []
        for aggregated_row in aggregator.values():
            aggregated_row = dict(aggregated_row)
            costs_rows.append([aggregated_row.get(col_name, 0.0) for col_name in costs_columns])

        monthly_costs_columns = copy.deepcopy(monthly_detail_columns)
        monthly_costs_columns.extend(self.monthly_columns())
        monthly_costs_columns.extend(self.monthly_columns(prefix='delta_'))
        monthly_costs_columns.extend(self.monthly_columns(prefix='target_'))
        monthly_costs_columns.extend(self.monthly_columns(mid_name='provider_'))
        monthly_costs_columns.extend(self.monthly_columns(prefix='delta_', mid_name='provider_'))
        monthly_costs_columns.extend(self.monthly_columns(prefix='target_', mid_name='provider_'))
        monthly_costs_columns.extend(self.monthly_columns(mid_name='network_'))
        monthly_costs_columns.extend(self.monthly_columns(prefix='delta_', mid_name='network_'))
        monthly_costs_columns.extend(self.monthly_columns(prefix='target_', mid_name='network_'))
        monthly_costs_rows = []
        for monthly_aggregated_row in monthly_aggregator.values():
            monthly_aggregated_row = dict(monthly_aggregated_row)
            monthly_costs_rows.append([monthly_aggregated_row.get(col_name, 0.0) for col_name in monthly_costs_columns])

        based_monthly_costs_columns = copy.deepcopy(monthly_costs_columns)
        based_monthly_aggregator = copy.deepcopy(monthly_aggregator)
        self.add_base_costs(based_monthly_aggregator)
        based_monthly_costs_rows = []
        for based_monthly_aggregated_row in based_monthly_aggregator.values():
            based_monthly_aggregated_row = dict(based_monthly_aggregated_row)
            based_monthly_costs_rows.append(
                [based_monthly_aggregated_row.get(col_name, 0.0) for col_name in based_monthly_costs_columns]
            )

        yoy_monthly_costs_columns = copy.deepcopy(based_monthly_costs_columns)
        yoy_monthly_aggregator = {
            k: v for (k, v) in copy.deepcopy(based_monthly_aggregator).items() if v['order_type'] != 'HWR_SERVER_ORDER'
        }
        self.distribute_yoy_capex(yoy_monthly_aggregator, yoy_monthly_costs_columns)
        yoy_monthly_costs_rows = []
        for yoy_monthly_aggregated_row in yoy_monthly_aggregator.values():
            yoy_monthly_aggregated_row = dict(yoy_monthly_aggregated_row)
            yoy_monthly_costs_rows.append(
                [yoy_monthly_aggregated_row.get(col_name, 0.0) for col_name in yoy_monthly_costs_columns]
            )
        return (
            costs_columns,
            costs_rows,
            monthly_costs_columns,
            monthly_costs_rows,
            based_monthly_costs_columns,
            based_monthly_costs_rows,
            yoy_monthly_costs_columns,
            yoy_monthly_costs_rows,
            costs_sort_order,
        )

    def _prepare_cost_pivot(self, filter_mode=''):
        def add_cost_field(field, caption, num_format, color):
            cost_pivot['value_fields'].append(field)
            cost_pivot['value_captions'].append(caption)
            cost_pivot['value_formats'].append(num_format)
            cost_pivot['value_colors'].append(color)

        cost_pivot = {
            'source_sheet': 'raw_costs',
            'sheet_color': 'White',
            'column_fields': [],
            'row_fields': ['ABC_SERVICE_1', 'ABC_SERVICE_2', 'SUMMARY', 'SUBTICKET_URL'],
            'value_fields': [],
            'value_captions': [],
            'value_formats': [],
            'value_colors': [],
            'collapse_columns': [],
            'formula_columns': [],
            'formula_caption': [],
        }

        if filter_mode == 'network':
            add_cost_field('cost_network_total (₽/m)', 'demand (₽/m)', '#,##0', 'LightTurquoise')
            add_cost_field('delta_cost_network_total (₽/m)', 'cut (₽/m)', '#,##0', 'Gray40')
            add_cost_field('target_cost_network_total (₽/m)', 'target (₽/m)', '#,##0', 'PowderBlue')
        elif filter_mode == 'provider':
            add_cost_field('cost_provider_total (₽/m)', 'demand (₽/m)', '#,##0', 'LightTurquoise')
            add_cost_field('delta_cost_provider_total (₽/m)', 'cut (₽/m)', '#,##0', 'Gray40')
            add_cost_field('target_cost_provider_total (₽/m)', 'target (₽/m)', '#,##0', 'PowderBlue')
        elif filter_mode == 'provider_gpu':
            add_cost_field('cost_provider_gpu_total (₽/m)', 'demand (₽/m)', '#,##0', 'LightTurquoise')
            add_cost_field('delta_cost_provider_gpu_total (₽/m)', 'cut (₽/m)', '#,##0', 'Gray40')
            add_cost_field('target_cost_provider_gpu_total (₽/m)', 'target (₽/m)', '#,##0', 'PowderBlue')
        elif filter_mode == 'provider_no_gpu':
            add_cost_field('cost_provider_no_gpu_total (₽/m)', 'demand (₽/m)', '#,##0', 'LightTurquoise')
            add_cost_field('delta_cost_provider_no_gpu_total (₽/m)', 'cut (₽/m)', '#,##0', 'Gray40')
            add_cost_field('target_cost_provider_no_gpu_total (₽/m)', 'target (₽/m)', '#,##0', 'PowderBlue')
        else:
            add_cost_field('cost_total (₽/m)', 'demand (₽/m)', '#,##0', 'LightTurquoise')
            add_cost_field('delta_cost_total (₽/m)', 'cut (₽/m)', '#,##0', 'Gray40')
            add_cost_field('target_cost_total (₽/m)', 'target (₽/m)', '#,##0', 'PowderBlue')

        start_index = 8
        for provider, provider_data in sorted(self.costs_sort_order.items(), key=lambda x: sum(x[1].values()), reverse=True):
            if filter_mode == 'network' and provider != 'strm.compute':
                continue
            if filter_mode == 'provider' and provider == 'strm.compute':
                continue
            if filter_mode == 'provider_gpu' and provider not in ('yt.gpu', 'yp.gpu'):
                continue
            if filter_mode == 'provider_no_gpu' and provider in ('yt.gpu', 'yp.gpu', 'strm.compute'):
                continue

            fixed_provider_name = provider.replace('.compute', '')
            add_cost_field('cost_{} (₽/m)'.format(provider), fixed_provider_name, '#,##0', 'LightTurquoise')
            add_cost_field('delta_cost_{} (₽/m)'.format(provider), '{} cut'.format(fixed_provider_name),
                           '#,##0', 'Gray40')
            add_cost_field('target_cost_{} (₽/m)'.format(provider), '{} target'.format(fixed_provider_name),
                           '#,##0', 'PowderBlue')

            offset = len(provider_data.items()) * 3
            start_col_name = excel_colnum_string(start_index)
            end_col_name = excel_colnum_string(start_index + offset - 1)
            cost_pivot['collapse_columns'].append('{}:{}'.format(start_col_name, end_col_name))
            start_index += (offset + 3)

            for res_data, _ in sorted(provider_data.items(), key=lambda x: x[1], reverse=True):
                sub_provider, res_name = res_data
                fixed_sub_provider_name = sub_provider.replace('.compute', '')
                fixed_res_name = res_name.replace('_segmented', '')

                add_cost_field('cost_{}_{} (₽/m)'.format(sub_provider, res_name),
                               '{} ({})'.format(fixed_res_name, fixed_sub_provider_name), '#,##0', 'White')
                add_cost_field('delta_cost_{}_{} (₽/m)'.format(sub_provider, res_name),
                               '{} cut ({})'.format(fixed_res_name, fixed_sub_provider_name), '#,##0', 'Gray25')
                add_cost_field('target_cost_{}_{} (₽/m)'.format(sub_provider, res_name),
                               '{} target ({})'.format(fixed_res_name, fixed_sub_provider_name), '#,##0', 'PaleBlue')
        return cost_pivot

    def _prepare_monthly_cost_pivot(self, prefix='', mid_name=''):
        def add_monthly_cost_field(field, caption, num_format, color):
            monthly_cost_pivot['value_fields'].append(field)
            monthly_cost_pivot['value_captions'].append(caption)
            monthly_cost_pivot['value_formats'].append(num_format)
            monthly_cost_pivot['value_colors'].append(color)

        monthly_cost_pivot = {
            'source_sheet': 'monthly_raw_costs',
            'sheet_color': 'White',
            'column_fields': [],
            'row_fields': ['ABC_SERVICE_1', 'ABC_SERVICE_2', 'SUMMARY', 'SUBTICKET_URL'],
            'value_fields': [],
            'value_captions': [],
            'value_formats': [],
            'value_colors': [],
            'collapse_columns': [],
            'formula_columns': [],
            'formula_caption': [],
        }
        add_monthly_cost_field('{}cost_{}yearly (₽/m)'.format(prefix, mid_name), 'За год', '#,##0', 'White')
        add_monthly_cost_field('{}cost_{}monthly_01 (₽/m)'.format(prefix, mid_name), 'Янаварь', '#,##0', 'White')
        add_monthly_cost_field('{}cost_{}monthly_02 (₽/m)'.format(prefix, mid_name), 'Февраль', '#,##0', 'White')
        add_monthly_cost_field('{}cost_{}monthly_03 (₽/m)'.format(prefix, mid_name), 'Март', '#,##0', 'White')
        add_monthly_cost_field('{}cost_{}monthly_04 (₽/m)'.format(prefix, mid_name), 'Апрель', '#,##0', 'White')
        add_monthly_cost_field('{}cost_{}monthly_05 (₽/m)'.format(prefix, mid_name), 'Май', '#,##0', 'White')
        add_monthly_cost_field('{}cost_{}monthly_06 (₽/m)'.format(prefix, mid_name), 'Июнь', '#,##0', 'White')
        add_monthly_cost_field('{}cost_{}monthly_07 (₽/m)'.format(prefix, mid_name), 'Июль', '#,##0', 'White')
        add_monthly_cost_field('{}cost_{}monthly_08 (₽/m)'.format(prefix, mid_name), 'Август', '#,##0', 'White')
        add_monthly_cost_field('{}cost_{}monthly_09 (₽/m)'.format(prefix, mid_name), 'Сентябрь', '#,##0', 'White')
        add_monthly_cost_field('{}cost_{}monthly_10 (₽/m)'.format(prefix, mid_name), 'Октябрь', '#,##0', 'White')
        add_monthly_cost_field('{}cost_{}monthly_11 (₽/m)'.format(prefix, mid_name), 'Ноябрь', '#,##0', 'White')
        add_monthly_cost_field('{}cost_{}monthly_12 (₽/m)'.format(prefix, mid_name), 'Декабрь', '#,##0', 'White')

        return monthly_cost_pivot

    def prepare_yoy_monthly_pivot(self, suffix=''):
        return {
            'source_sheet': 'monthly_raw_costs',
            'sheet_color': 'White',
            'column_fields': [],
            'row_fields': ['ABC_SERVICE_1', 'ABC_SERVICE_2', 'SUMMARY', 'SUBTICKET_URL'],
            'value_fields': [
                'cost{}_monthly_01 (₽/m)'.format(suffix),
                'cost{}_monthly_12 (₽/m)'.format(suffix),
                'dec_bill_yoy',
                'dec_bill_target_yoy',
                'target_cost{}_monthly_12 (₽/m)'.format(suffix),
                'dec_demand_yoy',
                'dec_bill_demand_yoy'
            ],
            'value_captions': [
                'Dec 2021 Bill',
                'Dec 2022 Bill',
                'Bill YoY, %',
                'Dec Bill YoY Target, %',
                'Dec 2022 Target Bill',
                'Dec 2022 Demand',
                'Dec 2022 Demand Target'
            ],
            'value_formats': ['#,##0.0', '#,##0.0', '0.00%', '0.00%', '#,##0.0', '#,##0.0', '#,##0.0'],
            'value_colors': ['White', 'White', 'White', 'White', 'White', 'White', 'White'],
            'collapse_columns': [],
            'formula_columns': [
                "(('cost{0}_monthly_12 (₽/m)' - 'cost{0}_monthly_01 (₽/m)') / 'cost{0}_monthly_01 (₽/m)')".format(suffix),
                "('cost{0}_monthly_12 (₽/m)' - 'cost{0}_monthly_01 (₽/m)')".format(suffix),
                "(('target_cost{0}_monthly_12 (₽/m)' - 'cost{0}_monthly_01 (₽/m)') / 'cost{0}_monthly_01 (₽/m)')".format(suffix),
                "('target_cost{0}_monthly_12 (₽/m)' - 'cost{0}_monthly_01 (₽/m)')".format(suffix),
            ],
            'formula_caption': [
                'dec_bill_yoy',
                'dec_demand_yoy',
                'dec_bill_target_yoy',
                'dec_bill_demand_yoy',
            ],
        }

    def prepare_compact_total_rows(self):
        provider_weight_tops = defaultdict(float)
        provider_weight_tops_res_name = {}
        provider_res_weight_tops = defaultdict(dict)
        for provider, provider_data in self.print_totals.items():
            for res_name, res_data in provider_data.items():
                weight = res_data.get('{}_{} (w)'.format(provider, res_name), 0.0)
                provider_res_weight_tops[provider][res_name] = weight
                if weight > provider_weight_tops[provider]:
                    provider_weight_tops[provider] = weight
                    provider_weight_tops_res_name[provider] = res_name

        # per dc weights
        self.compact_totals_columns.append('PROVIDER')
        self.compact_totals_columns.append('DC')
        self.compact_totals_columns.append('provider_dc_max_res_weight')
        for provider, res_name in provider_weight_tops_res_name.items():
            for dc, data in self.print_per_dc_totals[provider][res_name].items():
                new_row = {
                    'PROVIDER': provider,
                    'DC': dc,
                    'provider_dc_max_res_weight': data['{}_{}_{} (w)'.format(provider, res_name, dc)],
                }
                self.compact_totals_rows.append(new_row)

        # costs
        provider_costs = defaultdict(int)
        for row in self.initial_data:
            provider = row['details']['PROVIDER']
            for user_res_row in row.get('user_resources') or []:
                provider_costs[provider] += user_res_row['cost']
        self.compact_totals_columns.append('provider_cost')
        for provider, cost in provider_costs.items():
            new_row = {
                'PROVIDER': provider,
                'provider_cost': cost,
            }
            self.compact_totals_rows.append(new_row)

        return provider_res_weight_tops

    def prepare_pivot_data(self):
        def add_pivot_field(field, caption, num_format, color, provider_key):
            pivot_headers[provider_key]['value_fields'].append(field)
            pivot_headers[provider_key]['value_captions'].append(caption)
            pivot_headers[provider_key]['value_formats'].append(num_format)
            pivot_headers[provider_key]['value_colors'].append(color)

        based_pivot_headers = OrderedDict()
        based_row_fields = ['ABC_SERVICE_1', 'ABC_SERVICE_2', 'PROVIDER', 'SUMMARY', 'SUBTICKET_URL']
        based_pivot_headers['provider_monthly_costs'] = copy.deepcopy(self.provider_monthly_cost_pivot)
        based_pivot_headers['provider_monthly_costs']['row_fields'] = based_row_fields
        based_pivot_headers['network_monthly_costs'] = copy.deepcopy(self.network_monthly_cost_pivot)
        based_pivot_headers['network_monthly_costs']['row_fields'] = based_row_fields
        based_pivot_headers['total_monthly_costs'] = copy.deepcopy(self.monthly_cost_pivot)
        based_pivot_headers['total_monthly_costs']['row_fields'] = based_row_fields
        based_pivot_headers['provider_yoy_costs'] = self.prepare_yoy_monthly_pivot(suffix='_provider')
        based_pivot_headers['total_yoy_costs'] = self.prepare_yoy_monthly_pivot()

        yoy_pivot_headers = copy.deepcopy(based_pivot_headers)
        yoy_pivot_headers['total_yoy_costs']['value_captions'].extend(['2022 CAPEX', '2022 CAPEX Target'])
        yoy_pivot_headers['total_yoy_costs']['value_fields'].extend(['capex_demand', 'capex_demand_target'])
        yoy_pivot_headers['total_yoy_costs']['value_formats'].extend(['#,##0.0', '#,##0.0'])
        yoy_pivot_headers['total_yoy_costs']['value_colors'].extend(['White', 'White'])

        yoy_pivot_headers['provider_yoy_costs']['value_captions'].extend(['2022 CAPEX', '2022 CAPEX Target'])
        yoy_pivot_headers['provider_yoy_costs']['value_fields'].extend(['capex_provider_demand', 'capex_provider_demand_target'])
        yoy_pivot_headers['provider_yoy_costs']['value_formats'].extend(['#,##0.0', '#,##0.0'])
        yoy_pivot_headers['provider_yoy_costs']['value_colors'].extend(['White', 'White'])

        pivot_headers = OrderedDict()
        pivot_headers['providers_costs'] = self.provider_cost_pivot
        pivot_headers['providers_costs']['sheet_color'] = 'LightGreen'
        pivot_headers['providers_gpu_costs'] = self.provider_gpu_cost_pivot
        pivot_headers['providers_gpu_costs']['sheet_color'] = 'LightGreen'
        pivot_headers['providers_no_gpu_costs'] = self.provider_no_gpu_cost_pivot
        pivot_headers['providers_no_gpu_costs']['sheet_color'] = 'LightGreen'
        pivot_headers['providers_costs_flat'] = copy.deepcopy(self.provider_cost_pivot)
        pivot_headers['providers_costs_flat']['row_fields'] = ['SUMMARY', 'SUBTICKET_URL']
        pivot_headers['network_costs'] = self.network_cost_pivot
        pivot_headers['network_costs']['sheet_color'] = 'LightBlue'
        pivot_headers['total_costs'] = self.cost_pivot
        pivot_headers['total_costs']['sheet_color'] = 'LightLavender'
        pivot_headers['total_costs_flat'] = copy.deepcopy(self.cost_pivot)
        pivot_headers['total_costs_flat']['row_fields'] = ['SUMMARY', 'SUBTICKET_URL']
        pivot_headers['provider_monthly_costs'] = self.provider_monthly_cost_pivot
        pivot_headers['provider_monthly_cut'] = self.provider_monthly_delta_cost_pivot
        pivot_headers['provider_monthly_target'] = self.provider_monthly_target_cost_pivot
        pivot_headers['network_monthly_costs'] = self.network_monthly_cost_pivot
        pivot_headers['total_monthly_costs'] = self.monthly_cost_pivot

        pivot_headers['provider_dc'] = {
            'source_sheet': 'raw_total',
            'column_fields': ['DC'],
            'row_fields': ['PROVIDER'],
            'value_fields': ['provider_dc_max_res_weight'],
            'value_captions': [],
            'value_formats': ["#,##0.0"],
            'value_colors': ['White'],
            'collapse_columns': [],
            'formula_columns': [],
            'formula_caption': [],
        }
        for provider, _ in sorted(self.per_provider_sort_res.items(), key=lambda x: sum(x[1].values()), reverse=True):
            pivot_headers[provider] = {
                'source_sheet': 'raw_{}'.format(provider),
                'column_fields': [],
                'row_fields': ['ABC_SERVICE_1', 'ABC_SERVICE_2', 'SUMMARY', 'SUBTICKET_URL'],
                'value_fields': [],
                'value_captions': [],
                'value_formats': [],
                'value_colors': [],
                'collapse_columns': [],
                'formula_columns': [],
                'formula_caption': [],
            }
            cur_color = None
            color_carousel = {
                'White': 'Gray25',
                'Gray25': 'LightLavender',
                'LightLavender': 'White',
            }
            for col_name in self.per_provider_res_columns[provider]:
                cur_color = color_carousel.get(cur_color, 'White')
                col_caption = remove_prefix(col_name, '{}_'.format(provider))
                add_pivot_field(col_name, col_caption, "#,##0", cur_color, provider)
            if self.per_provider_res_columns[provider]:
                start_col_name = "E"
                end_col_name = excel_colnum_string(len(self.per_provider_res_columns[provider]) + 1)
                pivot_headers[provider]['collapse_columns'].append('{}:{}'.format(start_col_name, end_col_name))

            cur_color = None
            for i, res_data in enumerate(sorted(self.provider_res_weight_tops[provider].items(), key=lambda x: x[1], reverse=True)):
                res_name, weight = res_data
                fixed_res_name = res_name.replace('_segmented', '')
                unit = self.print_units[(provider, res_name)]
                if weight > 1.0:
                    cur_color = 'LightOrange' if cur_color != 'LightOrange' else 'Orange'
                else:
                    cur_color = 'LightGreen' if cur_color != 'LightGreen' else 'Green'

                add_pivot_field('{}_{} ({})'.format(provider, res_name, unit),
                                '{} ({})'.format(fixed_res_name, unit), "#,##0", cur_color, provider)
                add_pivot_field('{}_{}_limit ({})'.format(provider, res_name, unit),
                                '{}_limit ({})'.format(fixed_res_name, unit), "#,##0", cur_color, provider)
                add_pivot_field('{}_{} (w)'.format(provider, res_name),
                                '{} (w)'.format(fixed_res_name), "#,##0.0", cur_color, provider)
        return pivot_headers, based_pivot_headers, yoy_pivot_headers

    def _create_excel_file(self, target_path, excel_template_path=None, data_type=None):

        if excel_template_path:
            shutil.copy(excel_template_path, target_path)
        book = openpyxl.load_workbook(target_path, keep_vba=True)

        with pd.ExcelWriter(target_path, mode='a', if_sheet_exists='replace', engine='openpyxl') as writer:

            writer.book = book  # Hand over input workbook
            writer.sheets = dict((ws.title, ws) for ws in book.worksheets)  # Hand over worksheets
            writer.vba_archive = book.vba_archive  # Hand over VBA information

            if data_type == 'based':
                df = pd.DataFrame(self.based_monthly_costs_rows, columns=self.based_monthly_costs_columns)
                df.to_excel(writer, sheet_name='monthly_raw_costs', index=False)
            elif data_type == 'yoy':
                df = pd.DataFrame(self.yoy_monthly_costs_rows, columns=self.yoy_monthly_costs_columns)
                df.to_excel(writer, sheet_name='monthly_raw_costs', index=False)
            else:
                df = pd.DataFrame(self.costs_rows, columns=self.costs_columns)
                df.to_excel(writer, sheet_name='raw_costs', index=False)
                df = pd.DataFrame(self.monthly_costs_rows, columns=self.monthly_costs_columns)
                df.to_excel(writer, sheet_name='monthly_raw_costs', index=False)

                totals_rows = [[row.get(col_name) for col_name in self.compact_totals_columns] for row in self.compact_totals_rows]
                df = pd.DataFrame(totals_rows, columns=self.compact_totals_columns)
                df.to_excel(writer, sheet_name='raw_total', index=False)

                for provider, print_columns in self.per_provider_print_columns.items():
                    print_rows = self.per_provider_print_rows[provider]
                    df = pd.DataFrame(print_rows, columns=print_columns)
                    df.to_excel(writer, sheet_name='raw_{}'.format(provider), index=False)

            parameters_sheet = writer.sheets['parameters']
            parameters_sheet.delete_rows(1, 20)
            rows = []
            if data_type == 'based':
                pivot_data = self.based_pivot_data
            elif data_type == 'yoy':
                pivot_data = self.yoy_pivot_data
            else:
                pivot_data = self.pivot_data

            for pivot_name, pivot_data in reversed(pivot_data.items()):
                fixed_pivot_name = pivot_name.replace('.compute', '')
                source_sheet = pivot_data['source_sheet']
                if 'sheet_color' in pivot_data:
                    sheet_color = pivot_data['sheet_color']
                elif 'LightOrange' in pivot_data['value_colors']:
                    sheet_color = 'LightOrange'
                elif 'LightGreen' in pivot_data['value_colors']:
                    sheet_color = 'LightGreen'
                else:
                    sheet_color = 'White'
                rows.append([fixed_pivot_name, source_sheet, sheet_color, 'columns'] + pivot_data['column_fields'])
                rows.append(['', '', '', 'row_columns'] + pivot_data['row_fields'])
                rows.append(['', '', '', 'data_columns'] + pivot_data['value_fields'])
                rows.append(['', '', '', 'data_captions'] + pivot_data['value_captions'])
                rows.append(['', '', '', 'data_formats'] + pivot_data['value_formats'])
                rows.append(['', '', '', 'data_colors'] + pivot_data['value_colors'])
                rows.append(['', '', '', 'collapse_columns'] + pivot_data['collapse_columns'])
                rows.append(['', '', '', 'formula_columns'] + pivot_data['formula_columns'])
                rows.append(['', '', '', 'formula_caption'] + pivot_data['formula_caption'])

            max_row_len = max([len(f) for f in rows])
            for i in range(len(rows)):
                rows[i] += [''] * (max_row_len - len(rows[i]))
            rows = zip(*rows)

            for row in rows:
                parameters_sheet.append(row)

    def create_excel_file(self, target_path, excel_template_path=None):
        self._create_excel_file(target_path, excel_template_path=excel_template_path)

    def create_based_excel_file(self, target_path, excel_template_path=None):
        self._create_excel_file(target_path, excel_template_path=excel_template_path, data_type='based')

    def create_yoy_excel_file(self, target_path, excel_template_path=None):
        self._create_excel_file(target_path, excel_template_path=excel_template_path, data_type='yoy')

    def get_excel_file(self, excel_template_path=None):

        temp_excel_file = tempfile.NamedTemporaryFile(delete=False, suffix='.xlsm')
        temp_excel_file.close()
        temp_excel_path = temp_excel_file.name

        self.create_excel_file(self, temp_excel_path, excel_template_path=excel_template_path)

        with open(temp_excel_path, 'rb') as f:
            excel_data = f.read()
        os.unlink(temp_excel_path)

        return excel_data
