import requests
import urllib


class BaseApi(object):
    def __init__(self, host=None, token=None, robot_uid=None):
        self.token = token
        self.host = host
        self.session = requests.Session()
        if token:
            self.session.headers['Authorization'] = 'OAuth {}'.format(self.token)
        if robot_uid:
            self.session.headers['X-UID'] = robot_uid

    def _make_request(self, path, method='get', params=None, data=None):
        url = urllib.parse.urljoin(self.host, path)
        response = getattr(self.session, method)(url, params=params, json=data)
        if 400 <= response.status_code <= 500:
            errors = []
            try:
                response_data = response.json()
                errors = response_data.get('errors')
            except Exception:
                pass
            if errors:
                raise Exception(errors)
        response.raise_for_status()
        return response

    def _full_paged_data(self, path, next_key='nextPageToken', next_param='next'):
        result = []
        params = {}
        has_next = True
        while has_next:
            response = self._make_request(
                path=path,
                params=params
            )
            response_data = response.json()
            result.extend(response_data['items'])
            params[next_param] = response_data.get(next_key)
            has_next = True if params[next_param] else False
        return result

    def _full_paged_data_by_url(self, path):
        result = []
        has_next = True
        while has_next:
            response = self._make_request(path=path)
            response_data = response.json()
            result.extend(response_data['results'])
            path = response_data.get('next')
            has_next = True if path else False
        return result

    def get(self, path, method='get', params=None, data=None):
        return self._make_request(path=path, params=params, data=data)

    def put(self, path, method='put', params=None, data=None):
        return self._make_request(path=path, params=params, data=data)

    def post(self, path, method='post', params=None, data=None):
        return self._make_request(path=path, params=params, data=data)
