#!/usr/bin/env python
# coding=utf-8

import os
import json
import modifiers
import logging
import argparse
from collections import OrderedDict


def parse_args():

    parser = argparse.ArgumentParser(description='Cost price calculation')
    parser.add_argument('--settings', help='settings file', required=True, dest='settings')
    parser.add_argument('--data-provider', help='data provider', required=False, choices=['json', 'yt'], default='json',
                        dest='data_provider')
    parser.add_argument('--data-dir', help='data directory', required=True, dest='data_dir')

    return parser.parse_args()


def set_logging_level(verbose):
    logging.basicConfig(format=u'[%(levelname)-5s] [%(asctime)s] %(message)s')
    logging_level = logging.DEBUG if verbose else logging.INFO
    logging.getLogger().setLevel(level=logging_level)


def validate_settings(settings):
    # !TODO implements setting checks
    pass


def verify_data_changes(init_data, changed_data):
    # !TODO data checks
    pass


def fix_path(path):
    return os.path.realpath(os.path.expanduser(path))


def main():
    args = parse_args()

    set_logging_level(True)

    with open(args.settings, 'r') as f:
        settings = json.load(f, object_pairs_hook=OrderedDict)

    validate_settings(settings)

    data = []

    if "in_path" in settings:
        in_path = fix_path(os.path.join(args.data_dir, settings['in_path']))
        with open(in_path, 'r') as f:
            data = json.load(f, object_pairs_hook=OrderedDict)

    for operation in settings['pipeline']:

        class_name = operation['modifier']
        operation_parameters = operation.get('parameters', {})
        operation_parameters['data_dir'] = args.data_dir
        modifier_object = modifiers.by_name(class_name, data=data, **operation_parameters)

        if args.data_provider == 'json':
            dump_path = fix_path(os.path.join(args.data_dir, operation['out_path']))
            modifier_object.dump_json(dump_path)

        verify_data_changes(data, modifier_object.data)

        data = modifier_object.data


if __name__ == '__main__':
    main()
