#!/usr/bin/env python
# coding=utf-8

import os
import sys
import json
import csv
import logging
import argparse
from collections import OrderedDict


def set_logging_level(verbose):
    logging.basicConfig(format=u'[%(levelname)-5s] [%(asctime)s] %(message)s')
    logging_level = logging.DEBUG if verbose else logging.INFO
    logging.getLogger().setLevel(level=logging_level)


def parse_args():

    parser = argparse.ArgumentParser(description='fill elementary resources')
    parser.add_argument('--input', help='input data', required=True, dest='input_data')
    parser.add_argument('--data-dir', help='input data', required=True, dest='data_dir')

    return parser.parse_args()


def cast_number(line, key):
    val = line.get(key, 0) or 0
    return 0 if val == 'no_data' else int(val)


if __name__ == '__main__':

    set_logging_level(True)

    args = parse_args()

    path = os.path.join(args.data_dir, "tco_servers_v2.csv")

    tco_dict = {}
    with open(path, 'r') as f:
        reader = csv.DictReader(f)
        for row in reader:
            invnum = int(row['inv'])
            tco_dict[invnum] = {
                "Cores": {
                    "initial": cast_number(row, "Cores")
                },
                "NVME":  {
                    "initial": cast_number(row, "NVME")
                },
                "RAM":  {
                    "initial": cast_number(row, "RAM")
                },
                "HDD":  {
                    "initial": cast_number(row, "HDD")
                },
                "SSD":  {
                    "initial": cast_number(row, "SSD")
                }
            }

    with open(args.input_data, 'r') as f:
        data = json.load(f, object_pairs_hook=OrderedDict)

    for data_line in data:
        invnum = data_line["invnum"]
        if invnum in tco_dict:
            data_line["elementary_resources"] = tco_dict[data_line["invnum"]]
        else:
            logging.error('elementary resources not found for inv: {}'.format(invnum))

    sys.stdout.write(json.dumps(data))
