#!/usr/bin/env python
# coding=utf-8

import os
import sys
import json
import csv
import logging
import argparse
from collections import OrderedDict


def set_logging_level(verbose):
    logging.basicConfig(format=u'[%(levelname)-5s] [%(asctime)s] %(message)s')
    logging_level = logging.DEBUG if verbose else logging.INFO
    logging.getLogger().setLevel(level=logging_level)


def parse_args():

    parser = argparse.ArgumentParser(description='fill elementary resources costs')
    parser.add_argument('--input', help='input data', required=True, dest='input_data')
    parser.add_argument('--data-dir', help='input data', required=True, dest='data_dir')

    return parser.parse_args()


if __name__ == '__main__':

    set_logging_level(True)

    args = parse_args()

    path = os.path.join(args.data_dir, "tco_servers_v2.csv")

    cases = {
        "Servers D&A": "{}_D&A_4Y_ttl",
        "DC TCO": "{}_TCO"
    }
    tvm_add = 0.241
    ELEMENTARY_RESOURCES = ["Cores", "RAM", "HDD", "SSD", "NVME", "GPU"]

    tco_dict = {}
    with open(path, 'r') as f:
        reader = csv.DictReader(f)
        reader.fieldnames = [name.lower() for name in reader.fieldnames]
        for row in reader:
            invnum = int(row['inv'])
            elementary_resources_costs = OrderedDict()
            for resource in ELEMENTARY_RESOURCES:
                resource_costs = OrderedDict()
                for case, formatter in cases.iteritems():
                    name = formatter.format(resource).lower()
                    val = row[name] or 0.0
                    if val in ["-", "no_data"]:
                        val = 0.0
                    resource_costs[case] = float(val)
                    if case == "Servers D&A":
                        resource_costs["Servers D&A TVM"] = float(val) * tvm_add
                elementary_resources_costs[resource] = resource_costs
            tco_dict[invnum] = elementary_resources_costs

    with open(args.input_data, 'r') as f:
        data = json.load(f, object_pairs_hook=OrderedDict)

    for data_line in data:
        invnum = data_line["invnum"]
        if invnum in tco_dict:
            data_line["elementary_resources_costs"] = tco_dict[data_line["invnum"]]
        else:
            logging.error('elementary resources costs not found for inv: {}'.format(invnum))

    sys.stdout.write(json.dumps(data))
