#!/usr/bin/env python
# coding=utf-8

import os
import sys
import json
import logging
import argparse
import subprocess
from collections import OrderedDict


def set_logging_level(verbose):
    logging.basicConfig(format=u'[%(levelname)-5s] [%(asctime)s] %(message)s')
    logging_level = logging.DEBUG if verbose else logging.INFO
    logging.getLogger().setLevel(level=logging_level)


def parse_args():

    parser = argparse.ArgumentParser(description='fill elementary resources costs')
    parser.add_argument('--input', help='input data', required=True, dest='input_data')
    parser.add_argument('--data-dir', help='input data', required=True, dest='data_dir')

    return parser.parse_args()


def make_hosts_data():

    fields = [
        'dc',
        'invnum',
        'name',
        'model',
        'memory',
        'ssd',
        'disk',
        'switch',
        'queue',
        'rack',
        'gpu_models',
        'gpu_count',
        'ffactor',
        'net',
        'walle_tags',
    ]

    file_path = os.path.join(args.data_dir, 'HOSTS_DATA.csv')
    file_path = os.path.expanduser(file_path)
    file_path = os.path.realpath(file_path)

    gencfg_folder = os.path.realpath(os.path.join(os.path.dirname(__file__), '../../../../../gencfg'))
    commands = [
        'echo "{}" > {}'.format('\t'.join(fields), file_path),
        '{}/utils/common/dump_hostsdata.py -i {} >> {}'.format(gencfg_folder, ','.join(fields), file_path),
    ]

    for command in commands:
        logging.debug('launching {}'.format(command))
        subprocess.check_output(command, shell=True)


if __name__ == '__main__':

    set_logging_level(True)
    args = parse_args()

    with open(args.input_data, 'r') as f:
        data = json.load(f, object_pairs_hook=OrderedDict)

    inv_nums = set()
    for line in data:
        if not line.get('elementary_resources', {}):
            continue
        inv_nums.add(str(line['invnum']))

    # hosts_data_path = os.path.join(os.path.dirname(args.data_dir), "HOSTS_DATA.csv")
    hosts_data_path = os.path.join(args.data_dir, "HOSTS_DATA.csv")

    if not os.path.exists(hosts_data_path):
        make_hosts_data()

    hosts_data = []
    counter = 0
    with open(hosts_data_path, 'r') as f:
        counter += 1
        lines = f.readlines()[1:]
        for line in lines:
            line_segments = line.split('\t')
            if line_segments[1] in inv_nums:
                hosts_data.append(line)

    mode = "infra_only"
    gencfg_folder = os.path.realpath(os.path.join(os.path.dirname(__file__), '../../../../../gencfg'))
    command = gencfg_folder + '/calculate_yp_quota_for_hosts/calculate_yp_quota_for_hosts --dump --json'
    command += ' --discount-mode {}'.format(mode) if mode else ''
    process = subprocess.Popen(command, shell=True, stdin=subprocess.PIPE, stdout=subprocess.PIPE)
    stdout, stderr = process.communicate("".join(hosts_data))

    if process.returncode:
        print(stderr)
        raise OSError.ChildProcessError('calculate_yp_quota_for_hosts ended with code {}'.format(process.returncode))

    # !TODO debug output fix
    stdout_lines = stdout.split('\n')
    stdout_index = 0
    for i, line in enumerate(stdout_lines):
        if line == '{':
            stdout_index = i
            break
    stdout_lines = stdout_lines[stdout_index:]

    quota_data = json.loads('\n'.join(stdout_lines))

    QUOTA_RESOURCES = ["cpu", "memory", "hdd", "ssd", "io_hdd", "io_ssd"]
    QUOTA_MULTIPLIERS = {
        "hdd": 1024.0,
        "ssd": 1024.0
    }

    quota_dict = {}
    for row in quota_data["hosts"]:
        invnum = int(row["inv"])
        quota_line = OrderedDict()
        for res in QUOTA_RESOURCES:
            if not res in row["quota"]:
                logging.error("resource {} not defined for invnum {}".format(res, invnum))
            res_value = row["quota"].get(res, 0.0)
            res_value = res_value * QUOTA_MULTIPLIERS.get(res, 1.0)
            new_quota_line = OrderedDict({"initial": res_value})

            quota_line[res] = new_quota_line

        quota_dict[invnum] = quota_line

    for data_line in data:
        invnum = data_line["invnum"]
        if invnum in quota_dict:
            data_line["quota"] = quota_dict[data_line["invnum"]]
            segmentation_rules = data_line.get("segmentation", {})
            for res in QUOTA_RESOURCES:
                if res in segmentation_rules and res in data_line["quota"]:
                    discount_val = sum(data_line["quota"][res].values()) * (1.0 - segmentation_rules[res])
                    data_line["quota"][res]["segmentation_discount"] = -discount_val
        else:
            logging.error('quotas not found for inv: {}'.format(invnum))

    sys.stdout.write(json.dumps(data))
