#!/usr/bin/env python
# coding=utf-8

import os
import csv
import sys
import json
import logging
import argparse
import subprocess
from collections import OrderedDict


def set_logging_level(verbose):
    logging.basicConfig(format=u'[%(levelname)-5s] [%(asctime)s] %(message)s')
    logging_level = logging.DEBUG if verbose else logging.INFO
    logging.getLogger().setLevel(level=logging_level)


def parse_args():

    parser = argparse.ArgumentParser(description='fill elementary resources costs')
    parser.add_argument('--data-dir', help='input data', required=True, dest='data_dir')
    parser.add_argument('--input', help='input data', required=True, dest='input_data')

    return parser.parse_args()


def make_hosts_data():

    fields = [
        'dc',
        'invnum',
        'name',
        'model',
        'memory',
        'ssd',
        'disk',
        'switch',
        'queue',
        'rack',
        'gpu_models',
        'gpu_count',
        'ffactor',
        'net',
        'walle_tags',
    ]

    file_path = os.path.join(args.data_dir, 'HOSTS_DATA.csv')
    file_path = os.path.expanduser(file_path)
    file_path = os.path.realpath(file_path)

    gencfg_folder = os.path.realpath(os.path.join(os.path.dirname(__file__), '../../../../../gencfg'))
    commands = [
        'echo "{}" > {}'.format('\t'.join(fields), file_path),
        '{}/utils/common/dump_hostsdata.py -i {} >> {}'.format(gencfg_folder, ','.join(fields), file_path),
    ]

    for command in commands:
        logging.debug('launching {}'.format(command))
        subprocess.check_output(command, shell=True)


if __name__ == '__main__':

    set_logging_level(True)
    args = parse_args()

    with open(args.input_data, 'r') as f:
        data = json.load(f, object_pairs_hook=OrderedDict)

    hosts_data_path = os.path.join(args.data_dir, "HOSTS_DATA.csv")

    if not os.path.exists(hosts_data_path):
        make_hosts_data()

    gpu_count_dict = {}
    with open(hosts_data_path, 'r') as f:
        reader = csv.DictReader(f, delimiter='\t')
        for row in reader:
            if row['invnum'] in ('unknown',):
                continue
            gpu_count = row.get('gpu_count')
            if gpu_count:
                invnum = int(row['invnum'])
                gpu_count_dict[invnum] = int(gpu_count)


    for data_line in data:
        invnum = data_line["invnum"]

        if not data_line.get("elementary_resources_costs", {}):
            logging.error('elementary resources costs not found for inv: {}'.format(invnum))
            continue

        if invnum not in gpu_count_dict:
            logging.error('gpu quota not found for inv: {}'.format(invnum))
            continue

        gpu_count = gpu_count_dict[invnum]
        data_line["quota"] = {
            "gpu": {
                "initial": gpu_count
            }
        }

    sys.stdout.write(json.dumps(data))
