#!/usr/bin/env python
# coding=utf-8

import os
import sys
import csv
import json
import logging
import argparse
import subprocess
from collections import OrderedDict


def parse_args():

    parser = argparse.ArgumentParser(description='fill elementary resources costs')
    parser.add_argument('--input', help='input data', required=True, dest='input_data')
    parser.add_argument('--data-dir', help='input data', required=True, dest='data_dir')

    return parser.parse_args()


def set_logging_level(verbose):
    logging.basicConfig(format=u'[%(levelname)-5s] [%(asctime)s] %(message)s')
    logging_level = logging.DEBUG if verbose else logging.INFO
    logging.getLogger().setLevel(level=logging_level)


gpu_names_map = {
    'NVIDIA - GEFORCE GTX1080TI': "gpu_geforce_1080ti",
    'NVIDIA - GTX1080': "gpu_geforce_1080ti",
    "Asus - TURBO-GTX1080TI-11G": "gpu_geforce_1080ti",
    "Gigabyte - GV-N1080D5X-8GD-B": "gpu_geforce_1080ti",
    "Gigabyte - GV-N108TTURBO-11GD": "gpu_geforce_1080ti",
    "nVidia - GeForce GTX1080Ti (PG611)": "gpu_geforce_1080ti",
    "Palit - GTX1080Ti (NEB108T019LC-1021F)": "gpu_geforce_1080ti",
    "Zotac - ZT-P10810B-10B": "gpu_geforce_1080ti",
    "Zotac - ZT-P10810B-10P": "gpu_geforce_1080ti",
    "Gigabyte - GV-N1080NC-8GD": "gpu_geforce_1080ti",
    'TESLA K40': "gpu_tesla_k40",
    "NVIDIA-TeslaK40": "gpu_tesla_k40",
    'TESLA M40': "gpu_tesla_m40",
    "NVIDIA-TeslaM40": "gpu_tesla_m40",
    'NVIDIA - TESLA P40': "gpu_tesla_p40",
    "Nvidia - Tesla P40 (PG610)": "gpu_tesla_p40",
    'NVIDIA - TESLA V100': "gpu_tesla_v100",
    'V100-SXM2-32GB': "gpu_tesla_v100",
    "nVidia - Tesla V100-PCIE-16GB": "gpu_tesla_v100",
    "nVidia - Tesla V100-PCIE-32GB": "gpu_tesla_v100",
    "nVidia - Tesla V100-SXM2-32GB": "gpu_tesla_v100",
    'NVIDIA - A100': "gpu_tesla_a100",
    "nVidia - A100-40GB": "gpu_tesla_a100",
}
for el in gpu_names_map.values():
    gpu_names_map[el] = el


def get_segment(line, model):

    tags = line['walle_tags'].split(',')

    yt_cluster_prefix = "rtc.yt_cluster-"
    yt_cluster = [tag for tag in tags if tag.startswith(yt_cluster_prefix)]
    if yt_cluster:
        assert len(yt_cluster) == 1
        assert "yt" in tags
        yt_cluster = yt_cluster[0].split("-", 1)[1]
    else:
        yt_cluster = None

    segment = "unknown"
    if "rtc" in tags:
        gpu_tag = [tag for tag in tags if tag.startswith("rtc.gpu-")][0]
        if gpu_tag == "rtc.gpu-none":
            pass
            # logging.error("gpu project without gpu tag: {}".format(line))

        if gpu_tag != "rtc.gpu-none":
            segment = "yt.gpu" if "yt" in tags else "rtc.gpu"
        elif "yt" in tags:
            if yt_cluster == "arnold" and "runtime" in tags:
                segment = "yt-arnold-with-rtc"
            elif yt_cluster == "yt-gpu":
                segment = "yt.gpu"
            elif yt_cluster:
                segment = "yt.cluster." + yt_cluster
            elif "yp" in tags:
                segment = "yp-main"
            else:
                segment = "yt.unknown"
        elif "yp" in tags:
            segment = "rtc.compute.yp"
        elif "qloud" in tags:
            segment = "rtc.compute.qloud"
        elif "runtime" in tags:
            segment = "rtc.compute.gencfg"
        else:
            segment = "rtc.unsorted_tag"

    if segment in ["yt.gpu", "rtc.gpu"]:
        segment += "." + model if model else "unknown"

    return segment


def main():

    args = parse_args()

    set_logging_level(True)

    fields = ['dc',
              'invnum',
              'name',
              'model',
              'memory',
              'ssd',
              'disk',
              'switch',
              'queue',
              'rack',
              'gpu_models',
              'gpu_count',
              'ffactor',
              'net',
              'walle_tags',
              ]

    file_path = os.path.join(args.data_dir, 'HOSTS_DATA.csv')
    file_path = os.path.expanduser(file_path)
    file_path = os.path.realpath(file_path)

    gencfg_folder = os.path.realpath(os.path.join(os.path.dirname(__file__), '../../../../../gencfg'))
    commands = [
        'echo "{}" > {}'.format('\t'.join(fields), file_path),
        '{}/utils/common/dump_hostsdata.py -i {} >> {}'.format(gencfg_folder, ','.join(fields), file_path),
    ]

    for command in commands:
        logging.debug('launching {}'.format(command))
        subprocess.check_output(command, shell=True)

    data_dict = OrderedDict()

    with open(file_path, 'r') as f:
        reader = csv.DictReader(f, delimiter='\t')
        for row in reader:

            invnum = row['invnum']
            if invnum == 'unknown' or invnum == '':
                logging.error('unknown or empty line: {}'.format(row))
                continue

            invnum = int(invnum)
            fqdn = row['name']

            gpu_count = row['gpu_count']
            if gpu_count and not gpu_count.isdigit():
                logging.error("Invalid number of gpu cards for {}: {}".format(invnum, gpu_count))

            gpu_models = row['gpu_models'].split(",")
            gpu_name = gpu_models[0]
            model = gpu_names_map.get(gpu_name, gpu_name)

            if len(set(gpu_models)) > 1:
                logging.error("GPU host with multiple card types {}: {}".format(invnum, ",".join(sorted(gpu_models))))

            segment = get_segment(row, model)

            data_dict[int(invnum)] = {
                "invnum": int(invnum),
                "fqdn": fqdn,
                "segment": segment,
            }

    if args.input_data:
        with open(args.input_data, 'r') as f:
            data = json.load(f, object_pairs_hook=OrderedDict)
        for line in data:
            if line["invnum"] in data_dict:
                line["fqdn"] = data_dict[line["invnum"]]["fqdn"]
                line["segment"] = data_dict[line["invnum"]]["segment"]
    else:
        data = data_dict.values()

    sys.stdout.write(json.dumps(data))


if __name__ == '__main__':
    main()
