#!/usr/bin/env python
# coding=utf-8
import copy
import json
import argparse
from collections import OrderedDict, defaultdict


def parse_args():

    parser = argparse.ArgumentParser(description='Cost price calculation')
    parser.add_argument('--json-path', help='path to json file', required=True, dest='json_path')

    return parser.parse_args()


def summarize(data, res_name, segmentation_flag=False):

    counter_dict = OrderedDict()
    empty_elements = []

    for line in data:
        line_elementary_resources_costs = line.get(res_name, {})
        if len(line_elementary_resources_costs) == 0:
            empty_elements.append(line['invnum'])
        segmentation = line.get("segmentation", {})
        for res_type, values in line_elementary_resources_costs.iteritems():
            if res_type not in counter_dict:
                counter_dict[res_type] = OrderedDict()
            for cost_type, val in values.iteritems():
                val = val * segmentation.get(res_type, 1.0) if segmentation_flag else val
                if cost_type in counter_dict[res_type]:
                    counter_dict[res_type][cost_type] += val
                else:
                    counter_dict[res_type][cost_type] = val

    return counter_dict, empty_elements


def print_summary(counter_dict, empty_elements, header=''):

    print('=' * 20 + header.upper() + '=' * 20)
    total_dict = defaultdict(float)
    for res_type, values in counter_dict.iteritems():
        print('-' * 10 + res_type + '-' * 10)
        total_row = 0.0
        for cost_type, val in values.iteritems():
            total_row += val
            total_dict[cost_type] += val
            print(cost_type, val)
        print('total', total_row)
    print('-' * 10 + 'total' + '-' * 10)
    total_row = 0.0
    for cost_type, val in total_dict.iteritems():
        total_row += val
        print(cost_type, val)
    print('total', total_row)
    print('-' * 10 + 'empty ' + header + '-' * 10)
    print(len(empty_elements))


def print_price(quantity_dict, costs_dict_1, costs_dict_2=None, header=''):

    print('=' * 20 + header.upper() + '=' * 20)
    quantity_total = defaultdict(float)
    for res_type, values in quantity_dict.iteritems():
        quantity_total[res_type] += sum(values.values()) if isinstance(values, dict) else values

    total_costs_dict = copy.deepcopy(costs_dict_1)
    if costs_dict_2:
        for key, val in costs_dict_2.iteritems():
            total_costs_dict[key].update(val)

    for res_type, values in total_costs_dict.iteritems():
        if res_type in quantity_total:
            print('-' * 10 + res_type + '-' * 10)
            total_val = 0.0
            for cost_type, val in values.iteritems():
                total_val += val
                if not val:
                    price = 0.0
                elif not quantity_total[res_type]:
                    price = '<zero division>'
                else:
                    price = val / quantity_total[res_type]
                print(cost_type, price)
            if not total_val:
                total_price = 0.0
            elif not quantity_total[res_type]:
                total_price = '<zero division>'
            else:
                total_price = total_val / quantity_total[res_type]
            print('total', total_price)


def main():
    args = parse_args()
    with open(args.json_path, 'r') as f:
        data = json.load(f, object_pairs_hook=OrderedDict)

    # segment_filter = ('rtc.compute.', 'rtc.gpu.', 'yt.cluster.', 'yt.gpu.',)
    # segment_filter = ('yt.gpu.gpu_tesla_p40',)
    # data = [f for f in data if f['segment'].startswith(segment_filter)]

    segments = defaultdict(float)

    elementary_resources, empty_elementary_resources = summarize(data, 'elementary_resources')
    elementary_res_costs, empty_elementary_res_costs = summarize(data, 'elementary_resources_costs')
    quota, empty_quota = summarize(data, 'quota')
    quota_costs, empty_quota_costs = summarize(data, 'quota_costs')
    quota_cost_operations, empty_quota_cost_operations = summarize(data, 'quota_cost_operations')

    quota_tariff_costs = copy.deepcopy(quota_costs)
    # quota_tariff_costs = {k: v.update(quota_costs.get(k, {}) or {}) for (k, v) in quota_tariff_costs.iteritems()}
    for k, v in quota_tariff_costs.iteritems():
        v.update(quota_cost_operations.get(k, {}) or {})
    empty_quota_tariff_costs = copy.deepcopy(empty_quota_costs)
    empty_quota_tariff_costs.extend(empty_quota_cost_operations)

    for line in data:
        segments[line['segment']] += 1

    # segments
    print('=' * 20 + 'SEGMENTS' + '=' * 20)
    total_segments = 0.0
    for key in sorted(segments):
        total_segments += segments[key]
        print(key, segments[key])
    print('total', total_segments)

    print_summary(quota, empty_quota, 'elementary resources')
    print_summary(elementary_res_costs, empty_elementary_res_costs, 'elementary resources costs')
    print_price(elementary_resources, elementary_res_costs, header='elementary resources price')

    print_summary(quota, empty_quota, 'quota')
    print_summary(quota_costs, empty_quota_costs, 'quota costs')
    print_price(quota, quota_costs, header='quota price')

    print_summary(quota_tariff_costs, empty_quota_tariff_costs, 'quota tariff costs')
    print_price(quota, quota_costs, quota_cost_operations, header='quota tariffs')


if __name__ == '__main__':
    main()
