
import datetime
import argparse
import yt.wrapper as yt


def parse_args():

    parser = argparse.ArgumentParser(description='Put raw data to yt')
    parser.add_argument('--yt-cluster', help='yt cluster', required=True, dest='yt_cluster')
    parser.add_argument('--yt-root-path', help='yt root path', required=True, dest='yt_root_path')
    parser.add_argument('--date-string', help='date string', required=False, dest='date_string')

    return parser.parse_args()


def get_current_date_string():
    current_date = datetime.datetime.now()
    return current_date.strftime("%Y/%m/%d")


def create_cost_table(yt_client, table_path):
    yt_client.create("table", table_path,
                     ignore_existing=True,
                     attributes={
                         "schema":
                             [
                                 {"name": "date", "type": "datetime", "sort_order": "ascending"},
                                 {"name": "segment", "type": "string", "sort_order": "ascending"},
                                 {"name": "sku", "type": "string", "sort_order": "ascending"},
                                 {"name": "val_type", "type": "string"},
                                 {"name": "value", "type": "double"},
                             ]
                     }
                     )


def append_cost_data(yt_client, yt_root_path, total_table_name, date_string):

    table_path = yt_root_path + '/daily/' + date_string + '/' + total_table_name
    table = yt_client.read_table(table_path)

    dest_data = []
    for row in table:
        year, month, day = [int(f) for f in date_string.split('/')]
        dest_data.append({
            "date": int(datetime.datetime(year, month, day).strftime("%s")),
            "segment": row["segment"],
            "sku": row["sku"],
            "val_type": row["val_type"],
            "value": row["value"],
        })

    dest_table_path = '<append=true>'+ yt_root_path + '/total' + '/' + total_table_name

    create_cost_table(yt_client, dest_table_path)
    table = yt_client.write_table(dest_table_path, sorted(dest_data, key=lambda x: (x['date'], x['segment'], x['sku'])))


def create_segment_table(yt_client, table_path):
    yt_client.create("table", table_path,
                     ignore_existing=True,
                     attributes={
                         "schema":
                             [
                                 {"name": "date", "type": "datetime", "sort_order": "ascending"},
                                 {"name": "segment", "type": "string", "sort_order": "ascending"},
                                 {"name": "processed", "type": "int64"},
                                 {"name": "unprocessed", "type": "int64"},
                             ]
                     }
                     )


def append_segments_data(yt_client, yt_root_path, total_table_name, date_string):

    table_path = yt_root_path +  '/daily/' + date_string + '/' + total_table_name
    table = yt_client.read_table(table_path)

    dest_data = []
    for row in table:
        year, month, day = [int(f) for f in date_string.split('/')]
        dest_data.append({
            "date": int(datetime.datetime(year, month, day).strftime("%s")),
            "segment": row["segment"],
            "processed": row["processed"],
            "unprocessed": row["unprocessed"],
        })

    dest_table_path = '<append=true>' + yt_root_path + '/total/' + total_table_name

    create_segment_table(yt_client, dest_table_path)
    table = yt_client.write_table(dest_table_path, sorted(dest_data, key=lambda x: (x['date'], x['segment'])))


if __name__ == '__main__':

    args = parse_args()

    yt_cluster = args.yt_cluster
    root_path = args.yt_root_path
    client = yt.YtClient(yt_cluster)

    table_names = [
        'total_quota',
        'total_quota_costs',
        'total_quota_cost_operations'
    ]

    date_string = args.date_string if args.date_string else get_current_date_string()
    append_segments_data(client, root_path, 'total_segment_count', date_string)
    for table_name in table_names:
        append_cost_data(client, root_path, table_name, date_string)
