import requests
import datetime
import argparse
import math

import yt.wrapper as yt
from infra.capacity_planning.utils.hardware_utilization.lib.yt_tables import create_table, get_tariff


sensors = 'yt.scheduler.pools.resource_limits.gpu|yt.scheduler.pools.resource_usage.gpu'
clusters = ('hahn', 'arnold')
default_params = {
    'maxPoints': '25',
    'from': (datetime.date.today() - datetime.timedelta(days=2)).isoformat() + 'T15:00:00Z',
    'to': (datetime.date.today() - datetime.timedelta(days=1)).isoformat() + 'T15:00:00Z',
}


class SolomonApi:
    def __init__(self, token):
        self.api_url = 'http://solomon.yandex.net/api/v2'
        self.session = requests.Session()
        self.session.headers['Authorization'] = 'OAuth {}'.format(token)
        self.session.headers['Content-Type'] = 'text/plain'
        self.session.headers['Accept'] = 'application/json'

    def get_data(
        self, cluster, sensor=sensors, service="scheduler_pools",
        tree="gpu_tesla_v100_cloud",  pool="<Root>", params=None
    ):
        if params is None:
            params = default_params
        data = f'{{cluster={cluster}, service={service}, sensor="{sensor}", tree={tree}, pool="{pool}"}}'
        result = self.session.post(
            f'{self.api_url}/projects/yt/sensors/data',
            data=data, params=params
        )
        return result.json()['vector']


def arg_parse():
    parser = argparse.ArgumentParser()
    parser.add_argument('--token', type=str, default=None)
    return parser.parse_args()


def main():
    args = arg_parse()
    token = args.token
    cluster = 'hahn'
    yt_path = '<append=true>//home/capacity_planning/gpu_idle/gpu_pool_resources'
    yt_client = yt.YtClient(cluster)
    schema = [
        'date:datetime',
        'cluster:string',
        'sensor:string',
        'value:int64',
        'billing:double',
        'money_per_hour:double',
    ]
    create_table(client=yt_client, path=yt_path, schema=schema)

    billing = get_tariff(yt_client)['gpu_tesla_a100']
    result = []
    api = SolomonApi(token=token)

    for cluster in clusters:
        data = api.get_data(cluster=cluster)
        for sensor in data:
            for i, timest in enumerate(sensor['timeseries']['timestamps']):
                value = (sensor['timeseries']['values'][i])
                value = 0 if value == 'NaN' else math.floor(value)
                row = {
                    'date': timest//1000,
                    'cluster': cluster,
                    'sensor': sensor['timeseries']['labels']['sensor'],
                    'value': value,
                    'billing': billing,
                    'money_per_hour': value * billing / 30 / 24
                }
                result.append(row)
    yt_client.write_table(yt_path, result)


if __name__ == '__main__':
    main()
