import copy
import time
from collections import OrderedDict
from datetime import date, timedelta
import json
import subprocess
import requests
import argparse
import os
import sys

import yt.wrapper as yt
from infra.capacity_planning.library.python.abc_tree import AbcApi
from infra.capacity_planning.utils.hardware_utilization.lib.yt_tables import create_table
from infra.capacity_planning.utils.hardware_utilization.lib.golovan import golovan_request


GPU_PER_HOST = 8
ABC_URL = 'https://abc-back.yandex-team.ru/api/v4'
NANNY_URL = 'http://nanny.yandex-team.ru'


class ModdedAbcApi(AbcApi):

    def get_name(self, id_):
        resources_url = f'{self.api_url}/services/?id={id_}&fields=name'
        result = self.session.get(resources_url).json()
        return result['results'][0]['name']

    def get_names(self, ids):
        ids_str = ','.join(map(str, ids))
        resources_url = f'{self.api_url}/services/?id__in={ids_str}&fields=name,id'
        req = self.session.get(resources_url).json()
        req = req['results']
        result = {f['id']: {'name_ru': f['name']['ru'], 'name_en': f['name']['en']} for f in req}
        return result


class NannyApi:
    def __init__(self, nanny_api_url, token):
        self.api_url = nanny_api_url
        self.session = requests.Session()
        self.session.headers['Authorization'] = f'OAuth {token}'

    def get_id(self, nanny):
        resources_url = f'{self.api_url}/v2/services/{nanny}/info_attrs/'
        result = self.session.get(resources_url).json()
        return result['content']['abc_group']


def get_yesterday_month_string():
    today = date.today()
    yesterday = today - timedelta(days=1)
    return yesterday.strftime('%Y-%m')


def get_token():
    token_path = os.path.expanduser('~/.abc/token')
    if not os.path.exists(token_path):
        sys.stderr.write(
            'NO TOKEN found at ~/.abc/token\n'
            'get it at '
            'https://oauth.yandex-team.ru/authorize?response_type=token&client_id=23db397a10ae4fbcb1a7ab5896dc00f6\n'
        )
        exit(1)
    with open(token_path) as f:
        return f.read().strip()


def collect_and_put(client, path, abc_session, nanny_session):
    """
    Collect data from yp and golovan, then put it ti the table
    :return: 0
    """
    for dc in ('iva', 'vla', 'man', 'sas'):
        result = []
        yp_data = collect_from_yp(dc)
        abc_nanny_ids, abc_pod_set_ids, abc_ids = get_abc_ids(yp_data, dc, nanny_session)
        abc_names = abc_session.get_names(abc_ids)
        number_of_pod = len(yp_data)
        for i_pod, host in enumerate(yp_data, start=1):
            row = OrderedDict()
            row['dc'] = dc
            row['pod_id'] = host['pod_id']

            pod_set_id = host['pod_set_id']
            row['pod_set_id'] = pod_set_id
            row['abc_service_id_pod_set'] = abc_pod_set_ids[pod_set_id]

            nanny_service = host['nanny_service']
            row['nanny_service'] = nanny_service
            row['abc_group_nanny'] = abc_nanny_ids[nanny_service]

            row['abc_service_name_ru_pod_set'] = abc_names[row['abc_service_id_pod_set']]['name_ru']
            row['abc_service_name_en_pod_set'] = abc_names[row['abc_service_id_pod_set']]['name_en']
            row['abc_service_name_ru_nanny'] = abc_names[row['abc_group_nanny']]['name_ru']
            row['abc_service_name_en_nanny'] = abc_names[row['abc_group_nanny']]['name_en']

            abc_data = abc_session.service_tree[row['abc_service_id_pod_set']]
            row['abc_slug'] = abc_data['slug']

            vs_or_top_id = abc_session.get_service_vs_or_top_id(service_id=row['abc_service_id_pod_set'])
            vs_or_top = abc_session.service_tree[vs_or_top_id]
            row['vs_id'] = vs_or_top['id']
            row['vs_slug'] = abc_data['slug']
            row['vs_name'] = abc_data['name']

            row['host'] = host['host']
            row['gpu_resources'] = host['number_gpu']
            for dct in host['tags']:
                if dct['key'] == 'INSTANCE_TAG_CTYPE':
                    row['ctype'] = dct['value']
                elif dct['key'] == 'INSTANCE_TAG_ITYPE':
                    row['itype'] = dct['value']
                elif dct['key'] == 'INSTANCE_TAG_PRJ':
                    row['prj'] = dct['value']
            if row['gpu_resources'] > 0:
                row['gpu_model'] = host['gpu_cards'][0]['model']
            golovan_data = coollect_from_golovan(row, dc)
            print(f'{i_pod} from {number_of_pod} in {dc} pod_id = {row["pod_id"]}')
            if golovan_data:
                result.extend(golovan_data)

        client.write_table(path, result)


def collect_from_yp(dc):
    """
    collect data from yp
    :param dc: str, address name
    Collect data from yp , then put it ti the table
    :return: 0
    """
    command = ''.join(
        [
            'ya tool yp select pod ',
            '--filter \'NOT is_null([/spec/gpu_requests]) AND [/labels/deploy_engine] = "YP_LITE"\' '
            f'--address {dc} ',
            '--selector /meta/id ',
            '--selector /labels/nanny_service_id ',
            '--selector /spec/node_id ',
            '--selector /meta/pod_set_id ',
            '--selector /spec/gpu_requests ',
            '--selector /spec/iss/instances/0/properties ',
            '--format json --no-tabular',
        ]
    )
    data = json.loads(subprocess.check_output(command, shell=True))
    data_from_json = []
    interest = {'INSTANCE_TAG_PRJ', 'INSTANCE_TAG_CTYPE', 'INSTANCE_TAG_ITYPE'}

    for host in data:
        gpu_cards = copy.deepcopy(host[4])
        tags = []
        if host[5] is not None:
            tags = [f for f in host[5] if f['key'] in interest]
        data_from_json.append(
            {
                'pod_id': host[0],
                'nanny_service': host[1],
                'host': host[2],
                'pod_set_id': host[3],
                'number_gpu': len(host[4]),
                'gpu_cards': gpu_cards,
                'tags': tags
            }
        )

    return data_from_json


def coollect_from_golovan(row, dc):
    """
    collect data from golovan
    :param row: dict
    :param dc: str
    :return: result: dict, data from golovan
    """
    one_hour = 3600
    end_time = int(time.mktime(time.strptime(time.strftime('%Y %m %d'), '%Y %m %d'))) - one_hour
    start_time = int(end_time) - 23 * one_hour
    full_signals = []
    pod_id = row['pod_id']
    host_adr = f'{pod_id}.{dc}.yp-c.yandex.net'

    if 'itype' not in row:
        print('NO tags ', pod_id, dc)
        return None

    base_signal = 'itype={};prj={};ctype={}:aver({{}})'.format(row['itype'], row['prj'], row['ctype'])
    signals = {
        'gpu_util': 'portoinst-gpu_utilization_/dev/nvidia{}_tmmv',
        'gpu_mem': 'portoinst-gpu_memory_utilization_/dev/nvidia{}_tmmv',
        'gpu_temp': 'portoinst-gpu_temperature_/dev/nvidia{}_tmmv'
    }
    for id_card in range(GPU_PER_HOST):
        full_signals.extend(list(base_signal.format(v.format(id_card)) for v in signals.values()))

    req = golovan_request(host=host_adr, period=one_hour,
                          start_time=start_time, end_time=end_time,
                          signals=full_signals)

    result = []
    for data in req:
        new_row = row.copy()
        new_row['date'] = data[0]
        for id_card in range(GPU_PER_HOST):
            card_data = data[1]
            gpu_util_key = f'gpu_utilization_{id_card}'
            gpu_mem_key = f'gpu_memory_utilization_{id_card}'
            gpu_temp_key = f'gpu_temperature_{id_card}'
            new_row[gpu_util_key] = card_data[base_signal.format(signals['gpu_util'].format(id_card))]
            new_row[gpu_mem_key] = card_data[base_signal.format(signals['gpu_mem'].format(id_card))]
            new_row[gpu_temp_key] = card_data[base_signal.format(signals['gpu_temp'].format(id_card))]

            if new_row[gpu_util_key] is not None:
                new_row[gpu_util_key] = float(new_row[gpu_util_key])
            if new_row[gpu_mem_key] is not None:
                new_row[gpu_mem_key] = float(new_row[gpu_mem_key])
            if new_row[gpu_temp_key] is not None:
                new_row[gpu_temp_key] = float(new_row[gpu_temp_key])
        result.append(new_row)

    return result


def abc_id_pod_set(pod_set_id, dc):
    command = (
        f'ya tool yp get pod_set {pod_set_id} --address {dc} --selector /meta/account_id --format json --no-tabular'
    )
    data = json.loads(subprocess.check_output(command, shell=True))
    return int(data[0].removeprefix('abc:service:'))


def get_abc_ids(data, dc, nanny_session):
    abc_nanny_ids = {}
    abc_pod_set_ids = {}
    ids = set()
    for host in data:
        nanny_service = host['nanny_service']
        if nanny_service not in abc_nanny_ids:
            row_id = nanny_session.get_id(nanny_service)
            abc_nanny_ids[nanny_service] = row_id
            ids.add(row_id)
        pod_set_id = host['pod_set_id']
        if pod_set_id not in abc_pod_set_ids:
            row_id = abc_id_pod_set(pod_set_id, dc)
            abc_pod_set_ids[pod_set_id] = row_id
            ids.add(row_id)
    return abc_nanny_ids, abc_pod_set_ids, ids


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--token', type=str, default=None)
    args = parser.parse_args()
    token = args.token
    if not token:
        token = get_token()

    abc_session = ModdedAbcApi(token=token)
    abc_session.load_abc_service_tree()
    nanny_session = NannyApi(NANNY_URL, token)

    cluster = 'hahn'
    yt_path = f'<append=true>//home/capacity_planning/hardware_utilization/gpu_runtime/{get_yesterday_month_string()}'
    yt_client = yt.YtClient(cluster)
    schema = [
        'date:datetime',
        'pod_id:string',
        'nanny_service:string',
        'dc:string',
        'pod_set_id:string',
        'abc_service_id_pod_set:int64',
        'abc_service_name_ru_pod_set:string',
        'abc_service_name_en_pod_set:string',
        'abc_group_nanny:int64',
        'abc_service_name_ru_nanny:string',
        'abc_service_name_en_nanny:string',
        'abc_slug:string',
        'vs_id:int64',
        'vs_slug:string',
        'vs_name:string',
        'host:string',
        'gpu_model:string',
        'gpu_resources:int64',
        'ctype:string',
        'itype:string',
        'prj:string',
        ]
    for i in range(GPU_PER_HOST):
        schema.append(f'gpu_utilization_{i}:double')
        schema.append(f'gpu_memory_utilization_{i}:double')
        schema.append(f'gpu_temperature_{i}:double')

    create_table(client=yt_client, path=yt_path,
                 schema=schema, force=False,
                 ignore_existing=True)
    collect_and_put(yt_client, yt_path, abc_session, nanny_session)


if __name__ == '__main__':
    main()
