
import argparse
import sys
from collections import defaultdict

from infra.capacity_planning.utils.quota_mover.src.lib.constants import (
    ABC_URL_TEST,
    ABC_URL_PROD,
)
from infra.capacity_planning.utils.quota_mover.src.lib.exceptions import (
    AbcdException,
    ParameterRequiredError,
    ProviderNotFound,
    IncorrectStatusError,
)
from infra.capacity_planning.utils.quota_mover.src.lib.d_api import DApi
from infra.capacity_planning.utils.quota_mover.src.lib.quota_operations import (
    all_providers,
    create_provision,
    create_transfer_request,
    find_folder_account_id,
    fill_transfer_data_ids,
    transfers_from_parameters,
    get_service_folders,
    get_all_unit_ensemblies,
    get_segments_by_providers,
    get_all_units,
    convert_to_max_units,
)


def get_parser():
    parser = argparse.ArgumentParser()
    parser.add_argument('--list-providers', required=False, action='store_true')
    parser.add_argument('--list-resources', required=False, action='store_true')
    parser.add_argument('--list-segments', required=False, action='store_true')
    parser.add_argument('--folder-state', required=False, action='store_true')
    parser.add_argument('--move', required=False, action='store_true')
    parser.add_argument('--drop', required=False, action='store_true')
    parser.add_argument('--rise', required=False, action='store_true')

    parser.add_argument('--token', required=False)
    parser.add_argument('--test', required=False, action='store_true')
    parser.add_argument('--json', required=False)
    parser.add_argument('--provider', required=False)
    parser.add_argument('--segments', nargs='*', required=False)
    parser.add_argument('--from-service', required=False)
    parser.add_argument('--to-service', required=False)
    parser.add_argument('--from-folder', required=False)
    parser.add_argument('--to-folder', required=False)
    parser.add_argument('--from-account', required=False)
    parser.add_argument('--to-account', required=False)
    parser.add_argument('--resource', nargs='*', required=False)
    parser.add_argument('--comment', default='Quota migration by cli tool', required=False)
    return parser


def print_list_resources(d_client, provider_id, unit_ensembles):
    reduced_res = {}
    for res_type in d_client.provider_resource_types(provider_id):
        res_key = res_type['key']
        ensemble_id = res_type['unitsEnsembleId']
        reduced_res[res_key] = [unit_key for unit_key in unit_ensembles[ensemble_id]]
    res_name_len = max(len(res_name) for res_name in reduced_res)
    for res_name, units in sorted(reduced_res.items()):
        print('{:{width}} - {}'.format(res_name, ', '.join(units), width=res_name_len))


def print_list_segments(segments_by_providers):
    for provider_name, provider_data in segments_by_providers.items():
        print(f'{provider_name}:', end=' ')
        for segmentation_name, segmentation_data in provider_data.items():
            print(f'  {segmentation_name}: {", ".join([segment_data["name"] for segment_data in segmentation_data])}')


def move_quota(
    abc_url,
    d_client,
    rise,
    drop,
    provider,
    segments,
    from_service,
    to_service,
    from_folder,
    to_folder,
    from_account,
    to_account,
    resource,
    comment
):
    if not provider:
        raise ParameterRequiredError('--provider parameter required')
    if not segments:
        raise ParameterRequiredError('--segments parameter required')

    transfers = transfers_from_parameters(
        provider=provider,
        segments=segments,
        from_service=from_service,
        to_service=to_service,
        from_folder=from_folder,
        to_folder=to_folder,
        resource=resource
    )
    unit_ensembles = fill_transfer_data_ids(d_client, transfers)

    provider_id = transfers[0]['provider_id']
    provider_name = transfers[0]['provider']
    segments = transfers[0]['segments']
    segment_ids = transfers[0]['segment_ids']
    source_folder_name = transfers[0]['source_folder_name']
    target_folder_name = transfers[0]['target_folder_name']
    source_folder_id = transfers[0]['source_folder_id']
    target_folder_id = transfers[0]['target_folder_id']
    source_service_id = transfers[0]['source_service']
    target_service_id = transfers[0]['target_service']

    if rise:
        account_id = find_folder_account_id(d_client, provider_id, provider_name, source_folder_id, source_service_id,
                                            source_folder_name, segments, segment_ids, account_name=from_account)
        rise_provisions = create_provision(d_client, unit_ensembles, provider_id, source_folder_id, account_id,
                                           transfers, 'rise')
        if not rise_provisions['updatedProvisions']:
            print(f'Rising quotas [NOT_NEEDED], provider {provider_name}, service {source_service_id}')
        else:
            print(f'Rising quotas, provider {provider_name}, service {source_service_id}')
            d_client.provide_d_quota(source_folder_id, account_id, rise_provisions)

    transfer_data = create_transfer_request(d_client, source_folder_id, target_folder_id, transfers, comment)
    response = d_client.create_quota_transfer(transfer_data)
    response_data = response.json()
    status = response_data['status']
    quota_request_id = response_data['id']
    quota_request_url = f'{abc_url}/folders/transfers/{quota_request_id}'
    print(f'Moving quota request [{status}]: {quota_request_url}')

    if drop:
        # TODO при отрыве логики спуска/подъема - нужно будет учитывать что спуск/подъем может быть без переноса
        if status != 'APPLIED':
            text = (f'Request is in status - [{status}], '
                    f'drop is possible only for applications with the status [APPLIED]'
                    )
            raise IncorrectStatusError(text)
        account_id = find_folder_account_id(d_client, provider_id, provider_name, target_folder_id, target_service_id,
                                            target_folder_name, segments, segment_ids, account_name=to_account)
        drop_provisions = create_provision(d_client, unit_ensembles, provider_id, target_folder_id, account_id,
                                           transfers, 'drop')
        print(f'dropping the quota, provider {provider_name}, folder {target_service_id}')
        d_client.provide_d_quota(target_folder_id, account_id, drop_provisions)
        print('SUCCESS')


def update_max_len(new_row_part, max_length_array):
    for i, row_part in enumerate(new_row_part):
        if len(max_length_array) <= i:
            max_length_array.append(max(len(row_part), 1))
        else:
            max_length_array[i] = max(max_length_array[i], len(row_part))


def abcd(**kwargs):
    abc_url = ABC_URL_TEST if kwargs['test'] else ABC_URL_PROD
    d_client = DApi(token=kwargs['token'], test=kwargs['test'])

    if kwargs['list_providers']:
        for provider_row in sorted(d_client.providers(), key=lambda x: x['key']):
            print(provider_row['key'])
    elif kwargs['list_resources']:
        if not kwargs['provider']:
            raise ParameterRequiredError('--provider parameter required')

        all_d_providers = all_providers(d_client)
        if kwargs['provider'] not in all_d_providers:
            provider = kwargs['provider']
            exception_rows = [f'\nProvider "{provider}" was not found. Here is the list of available providers:\n']
            exception_rows.extend([f'    {provider}\n' for provider in all_d_providers])
            raise ProviderNotFound(''.join(exception_rows))

        provider_id = all_d_providers[kwargs['provider']]['id']
        unit_ensembles = get_all_unit_ensemblies(d_client)
        print_list_resources(d_client, provider_id, unit_ensembles)
    elif kwargs['list_segments']:
        providers = [kwargs['provider']] if kwargs['provider'] else None
        segments_by_providers = get_segments_by_providers(d_client, providers)
        print_list_segments(segments_by_providers)
    elif kwargs['folder_state']:
        if not kwargs['from_service'] and not kwargs['to_service']:
            raise ParameterRequiredError('--from-service or --to-service parameters required')
        service = kwargs['from_service'] or kwargs['to_service']
        service_folders = get_service_folders(d_client, [service])
        all_d_providers = all_providers(d_client)
        unit_names, unit_ids, unit_ensembles = get_all_units(d_client)

        for folder_name, folder_data in service_folders[service].items():
            folder_id = folder_data['id']
            print(f'{folder_name}:')
            folder_quota = d_client.get_folder_quota(folder_id)
            folder_quotas = defaultdict(list)
            for quota_line in folder_quota:
                provider_id = quota_line['providerId']
                folder_quotas[provider_id].append(quota_line)
            folder_accounts = d_client.get_folder_accounts(folder_id)
            provider_folder_accounts = defaultdict(list)
            for account_line in folder_accounts:
                provider_id = account_line['providerId']
                provider_folder_accounts[provider_id].append(account_line)
            for provider_row in sorted(all_d_providers.values(), key=lambda x: x['key'].lower()):
                provider_id = provider_row['id']
                provider_key = provider_row['key']
                if provider_id in folder_quotas:
                    if not folder_quotas[provider_id]:
                        continue
                    print(f'  {provider_key}:')
                    provider_rows = []
                    provider_row_max_lengths = []
                    provision_amounts = defaultdict(list)
                    for account_line in provider_folder_accounts[provider_id]:
                        account_id = account_line['id']
                        account_provisions = d_client.get_provisions(folder_id, account_id)
                        for single_provision in account_provisions:
                            resource_id = single_provision['resourceId']
                            provision_amounts[resource_id].append(single_provision)
                    for quota in folder_quotas[provider_id]:
                        resource_id = quota['resourceId']
                        resource_data = d_client.get_resource(provider_id, resource_id)
                        ensemble_id = resource_data['unitsEnsembleId']
                        resource_key = resource_data['key']
                        quota_amount, quota_unit = convert_to_max_units(unit_ensembles, ensemble_id,
                                                                        quota['quota'], quota['quotaUnitKey'])
                        balance_amount, balance_unit = convert_to_max_units(unit_ensembles, ensemble_id,
                                                                            quota['balance'], quota['balanceUnitKey'])
                        new_row = [
                            f'    {resource_key} - ',
                            f'{quota_amount} {quota_unit} ',
                            f'| {balance_amount} {balance_unit} ',
                            '',
                            '',
                        ]
                        provision_rows = provision_amounts.get(resource_id)
                        if not provision_rows:
                            update_max_len(new_row, provider_row_max_lengths)
                            provider_rows.append(new_row)
                        else:
                            for i, provision_row in enumerate(provision_rows):
                                provided_amount, provided_unit = convert_to_max_units(unit_ensembles, ensemble_id,
                                                                                      provision_row['provided'],
                                                                                      provision_row['providedUnitKey'])
                                alloc_amount, alloc_unit = convert_to_max_units(unit_ensembles, ensemble_id,
                                                                                provision_row['allocated'],
                                                                                provision_row['allocatedUnitKey'])
                                if i == 0:
                                    new_row[3] = f'| {provided_amount} {provided_unit} '
                                    new_row[4] = f'| {alloc_amount} {alloc_unit} '
                                else:
                                    new_row = [
                                        f'    {resource_key} - ',
                                        '',
                                        '',
                                        f'| {provided_amount} {provided_unit} ',
                                        f'| {alloc_amount} {alloc_unit} ',
                                    ]
                                update_max_len(new_row, provider_row_max_lengths)
                                provider_rows.append(new_row)

                    amount_units_row = [
                        '    ------',
                        'QUOTA ',
                        'FREE QUOTA ',
                        'PROVIDED ',
                        'ALLOCATED ',
                    ]
                    update_max_len(amount_units_row, provider_row_max_lengths)
                    provider_rows.append(amount_units_row)
                    for row_parts in sorted(provider_rows):
                        row = ''.join([
                            '{:{width}}'.format(part, width=provider_row_max_lengths[i])
                            for i, part in enumerate(row_parts)
                        ])
                        print(row)
    elif kwargs['move']:
        move_quota(
            abc_url,
            d_client,
            kwargs['rise'],
            kwargs['drop'],
            kwargs['provider'],
            kwargs['segments'],
            kwargs['from_service'],
            kwargs['to_service'],
            kwargs['from_folder'],
            kwargs['to_folder'],
            kwargs['from_account'],
            kwargs['to_account'],
            kwargs['resource'],
            kwargs['comment']
        )
    elif kwargs['provider']:
        all_d_providers = all_providers(d_client)
        if kwargs['provider'] not in all_d_providers:
            provider = kwargs['provider']
            exception_rows = [f'\nProvider "{provider}" was not found. Here is the list of available providers:\n']
            exception_rows.extend([f'    {provider}\n' for provider in all_d_providers])
            raise ProviderNotFound(''.join(exception_rows))
        else:
            print('Resources:\n')
            provider_id = all_d_providers[kwargs['provider']]['id']
            unit_ensembles = get_all_unit_ensemblies(d_client)
            print_list_resources(d_client, provider_id, unit_ensembles)
            print('\nSegments:\n')
            providers = [kwargs['provider']] if kwargs['provider'] else None
            segments_by_providers = get_segments_by_providers(d_client, providers)
            print_list_segments(segments_by_providers)
            exit(0)


def main():
    parser = get_parser()
    if len(sys.argv) < 2:
        parser.print_help()
        exit(1)
    args = parser.parse_args()
    try:
        abcd(**args.__dict__)
    except AbcdException as exc:
        print(f'{type(exc).__name__}: {exc}')
