import os
import sys
import requests
import urllib.parse
from retrying import retry
from infra.capacity_planning.utils.quota_mover.src.lib.constants import (
    D_URL_TEST,
    D_URL_PROD,
    D_CLIENT_ID_TEST,
    D_CLIENT_ID_PROD,
)

requests.packages.urllib3.disable_warnings()


def get_token(test=False):

    token_path = os.path.expanduser('~/.d/token')
    if 'D_TOKEN' in os.environ:
        return os.environ['D_TOKEN']
    elif os.path.exists(token_path):
        with open(token_path) as f:
            return f.read().strip()

    client_id = D_CLIENT_ID_TEST if test else D_CLIENT_ID_PROD
    sys.stderr.write(
        'NO TOKEN found at env variable D_TOKEN or in file ~/.d/token\n'
        'get it at https://oauth.yandex-team.ru/authorize?response_type=token&client_id={}\n'.format(client_id)
    )
    exit(1)


class DApi(object):
    def __init__(self, host=None, token=None, test=False):
        if token is None:
            token = get_token(test=test)

        if host:
            self.host = host
        elif test:
            self.host = D_URL_TEST
        else:
            self.host = D_URL_PROD
        self.session = requests.Session()
        self.session.verify = False
        self.session.headers['Authorization'] = 'OAuth {}'.format(token)

    @retry(stop_max_attempt_number=3)
    def _make_request(self, path, method='get', params=None, data=None):
        url = urllib.parse.urljoin(self.host, path)
        response = getattr(self.session, method)(url, params=params, json=data)
        if 400 <= response.status_code <= 500:
            errors = []
            try:
                response_data = response.json()
                errors = response_data.get('errors')
                field_errors = response_data.get('fieldErrors')
            except Exception:
                pass
            if errors:
                raise Exception(errors)
            if field_errors:
                raise Exception(field_errors)
        response.raise_for_status()
        return response

    def _full_paged_data(self, path, params=None):
        result = []
        if params is None:
            params = {}
        has_next = True
        while has_next:
            response = self._make_request(
                path=path,
                params=params
            )
            response_data = response.json()
            result.extend(response_data['items'])
            params['pageToken'] = response_data.get('nextPageToken')
            has_next = True if params['pageToken'] else False
        return result

    def providers(self):
        return self._full_paged_data(path='/api/v1/providers')

    def provider_by_key(self, key):
        provider_key = None
        for provider_row in self.providers():
            if provider_row['key'] == key:
                provider_key = provider_row['id']
                break
        return provider_key

    def units(self):
        return self._full_paged_data(path='/api/v1/unitsEnsembles')

    def provider_resources(self, provider_id):
        return self._full_paged_data(path=f'/api/v1/providers/{provider_id}/resources')

    def provider_resource_by_id(self, provider_id, resource_id):
        return self._make_request(path=f'/api/v1/providers/{provider_id}/resources/{resource_id}')

    def provider_segmentations(self, provider_id):
        return self._full_paged_data(path=f'/api/v1/providers/{provider_id}/resourcesDirectory/segmentations')

    def provider_segments(self, provider_id, segmentation_id):
        path = f'/api/v1/providers/{provider_id}/resourcesDirectory/segmentations/{segmentation_id}/segments'
        return self._full_paged_data(path=path)

    def provider_resource_types(self, provider_id):
        return self._full_paged_data(path=f'/api/v1/providers/{provider_id}/resourcesDirectory/resourceTypes')

    def provider_resources_by_segments(self, provider_id, resource_type_id, segment_ids):
        path = f'/api/v1/providers/{provider_id}/resources/filterByType/{resource_type_id}'
        params = {
            'segmentId': segment_ids,
            'withSegmentations': True,
        }
        return self._full_paged_data(path=path, params=params)

    def all_provider_resources(self, provider_key):
        provider_id = self.provider_by_key(provider_key)
        return self.provider_resources(provider_id)

    def get_resource(self, provider_id, resource_id):
        resource_data = self.provider_resource_by_id(provider_id, resource_id)
        return resource_data.json()

    def service_folders(self, abc_service_id):
        return self._full_paged_data(path=f'/api/v1/services/{abc_service_id}/folders')

    def create_quota_transfer(self, data):
        return self._make_request('/api/v1/transfers', method='post', data=data)

    def request_d_quota(self, data):
        return self._make_request('/api/v1/bulkQuotaMove', method='post', data=data)

    def get_provisions(self, folder_id, account_id):
        return self._full_paged_data(f'/api/v1/folders/{folder_id}/accounts/{account_id}/provisions')

    def get_provision(self, folder_id, account_id, resource_id):
        provisions = self.get_provisions(folder_id, account_id)
        for provision in provisions:
            if provision['resourceId'] == resource_id:
                return provision
        return None

    def get_folder_quota(self, folder_id):
        return self._full_paged_data(f'/api/v1/folders/{folder_id}/quotas')

    def provide_d_quota(self, folder_id, account_id, data):
        return self._make_request(f'/api/v1/folders/{folder_id}/accounts/{account_id}/_provide',
                                  method='post', data=data)

    def get_folder_accounts(self, folder_id):
        return self._full_paged_data(path=f'/api/v1/folders/{folder_id}/accounts')

    def get_folder_provider_accounts(self, folder_id, provider_id):
        return self._full_paged_data(path=f'/api/v1/folders/{folder_id}/providers/{provider_id}/accounts')

    def get_accounts_spaces(self, provider_id):
        return self._full_paged_data(path=f'/api/v1/providers/{provider_id}/accountsSpaces')
