
import json
from collections import defaultdict


def transfers_from_json(path):
    with open(path, 'r') as f:
        data = json.load(f)
        common_values = {key: val for key, val in data.items() if key != 'transfers'}
        transfers = data['transfers']
        for row in transfers:
            # !TODO do not overwrite existing keys
            row.update(common_values)
    return transfers


def transfers_from_parameters(provider, segments, from_service=None, to_service=None,
                              from_folder=None, to_folder=None, resource=None, pull=None, push=None):
    transfers = []
    resource = resource if resource else []
    for res_string in resource:
        res_string_components = res_string.split(':')
        if len(res_string_components) == 2 and res_string_components[1] == 'all':
            res, amount, unit = res_string_components[0], 'all', ''
        else:
            res, amount, unit = res_string_components

        if amount == 'all':
            amount = 'all'
        elif not amount.isdigit():
            raise Exception('Все передаваемые величины должны быть целочисленными')
        else:
            amount = int(amount)
        new_row = {
            'provider': provider,
            'segments': segments,
            'resource': res,
            'amount': amount,
            'unit': unit,
        }
        if from_service:
            new_row['source_service'] = from_service
        if to_service:
            new_row['target_service'] = to_service
        if from_folder:
            new_row['source_folder'] = from_folder
        if to_folder:
            new_row['target_folder'] = to_folder
        transfers.append(new_row)

    return transfers


def all_providers(d_client):
    all_d_providers = {}
    for provider_row in d_client.providers():
        all_d_providers[provider_row['key']] = {
            'id': provider_row['id'],
            'key': provider_row['key'],
            'abc_service_id': provider_row['abcServiceId'],
        }
    return all_d_providers


def get_all_units(d_client):
    unit_names = {}
    unit_ids = {}
    unit_ensemblies = defaultdict(list)
    for ensemblies in d_client.units():
        ensemble_id = ensemblies['id']
        for unit in ensemblies['units']:
            unit_id = unit['id']
            unit_key = unit['key']
            unit_ensemblies[ensemble_id].append(
                {
                    'id': unit_id,
                    'key': unit_key,
                    'base': unit['base'],
                    'power': unit['power'],
                }
            )
            unit_ids[unit_id] = unit_key
            unit_names[unit_key] = unit_id
    return unit_names, unit_ids, unit_ensemblies


def get_unit_multipliers(unit_ensemblies, ensemble_id, available_units=None):
    ok_units = [
        unit for unit in unit_ensemblies[ensemble_id]
        if available_units is None or unit['key'] in available_units
    ]
    base_mul, base_key, base_id = min([(unit['base'] ** unit['power'], unit['key'], unit['id']) for unit in ok_units])
    unit_multipliers = {}
    for unit in ok_units:
        unit_key = unit['key']
        mul = unit['base'] ** unit['power']
        unit_multipliers[unit_key] = (mul // base_mul, base_key, base_id)
    return unit_multipliers


def convert_to_max_units(unit_ensemblies, ensemble_id, amount, unit):
    multipliers = {unit['key']: unit['base'] ** unit['power'] for unit in unit_ensemblies[ensemble_id]}
    base_unit_multiplier = multipliers[unit]
    converted_amount, converted_unit = amount, unit
    if not amount:
        return converted_amount, converted_unit
    for mult_unit, unit_multiplier in multipliers.items():
        if unit_multiplier <= base_unit_multiplier:
            continue
        current_multiplier = unit_multiplier // base_unit_multiplier
        if amount % current_multiplier != 0:
            continue
        new_amount = amount // current_multiplier
        if new_amount < converted_amount:
            converted_amount, converted_unit = new_amount, mult_unit

    return converted_amount, converted_unit


def get_all_unit_ensemblies(d_client):
    ensemblie_units = defaultdict(dict)
    for unit_ensemblies in d_client.units():
        ensemblie_id = unit_ensemblies['id']
        for unit in unit_ensemblies['units']:
            ensemblie_units[ensemblie_id][unit['key']] = unit['id']
    return ensemblie_units


def unique_services_and_providers(all_d_providers, simple_transfers):
    services = set()
    providers = set()
    for row in simple_transfers:
        source_service = row.get('source_service') or all_d_providers[row['provider']]['abc_service_id']
        target_service = row.get('target_service') or all_d_providers[row['provider']]['abc_service_id']
        services.add(source_service)
        services.add(target_service)
        providers.add(row.get('provider'))
    return providers, services


def get_service_folders(d_client, services):
    service_folders = defaultdict(dict)
    for service in services:
        folders = d_client.service_folders(service)
        for folder in folders:
            service_folders[service][folder['displayName']] = {
                'id': folder['id'],
                'type': folder['folderType'],
                'name': folder['displayName'],
            }
    return service_folders


def get_resource_types(d_client, providers, all_d_providers):
    resource_types = defaultdict(dict)
    for provider_name in providers:
        provider_id = all_d_providers[provider_name]['id']
        for res_type in d_client.provider_resource_types(provider_id):
            resource_types[provider_name][res_type['key']] = res_type['id']
    return resource_types


def get_segments_by_keys(d_client, providers=None, all_d_providers=None):
    segments_by_providers = get_segments_by_providers(d_client, providers=providers, all_d_providers=all_d_providers)
    segments = defaultdict(list)
    for provider_name, provider_data in segments_by_providers.items():
        for segmentation_name, segmentation_data in provider_data.items():
            for segment_data in segmentation_data:
                segments[segment_data['name']].append(
                    segment_data
                )
    return segments


def get_segments_by_providers(d_client, providers=None, all_d_providers=None):
    if all_d_providers is None:
        all_d_providers = all_providers(d_client)
    if providers is None:
        providers = list(all_d_providers)

    provider_segments = defaultdict(lambda: defaultdict(list))
    for provider_name in providers:
        provider_id = all_d_providers[provider_name]['id']
        for segmentation in d_client.provider_segmentations(provider_id):
            for segment in d_client.provider_segments(provider_id, segmentation['id']):
                segmentation_name = segmentation['key']
                provider_segments[provider_name][segmentation_name].append(
                    {
                        'id': segment['id'],
                        'name': segment['key'],
                        'segmentation_id': segment['segmentationId'],
                        'segmentation_name': segmentation_name,
                        'provider_id': provider_id,
                        'provider_name': provider_name,
                    }
                )
    return provider_segments


def find_folder_account_id(d_client, provider_id, provider_name, folder_id, service_id, folder_name,
                           segments, segment_ids, account_name=None):

    accounts_spaces = d_client.get_accounts_spaces(provider_id)
    spaces_by_segments = {}
    for space in accounts_spaces:
        key = frozenset(segment['segment']['id'] for segment in space['segments'])
        spaces_by_segments[key] = space['id']

    segment_key = frozenset(segment_ids)
    space_id = spaces_by_segments.get(segment_key)

    account_data = d_client.get_folder_provider_accounts(folder_id, provider_id)
    if not account_data:
        exception_rows = [
            f'\nУ фолдера {folder_name} в сервисе {service_id}, ',
            f'нет привязанных акаунтов в провайдере {provider_name}\n',
        ]
        raise Exception(''.join(exception_rows))

    accounts = defaultdict(dict)
    for row in account_data:
        current_space_id = row.get('accountsSpaceId')
        external_key = row.get('externalKey')
        accounts[external_key][current_space_id] = row['id']

    account_id = None
    if account_name:
        space_acounts = accounts.get(account_name)
        account_id = space_acounts.get(space_id)
    elif len(accounts) == 1 and len(list(accounts.values())[0]):
        single_account = list(accounts.values())[0]
        account_id = list(single_account.values())[0]

    if not account_id:
        exception_rows = [
            f'\nПровайдер - {provider_name}, фолдер - {folder_name}, сервис - {service_id}\n',
        ]
        if account_name:
            exception_rows.append(f'Не удалось найти акаунт - {account_name}\n')
        else:
            exception_rows.append('К фолдеру привязано более одного акаунта, нужно явно указать акаунт\n')
        exception_rows.append('Доступные акаунты:\n')
        exception_rows.extend([f'{account}\n' for account in accounts])
        raise Exception(''.join(exception_rows))

    return account_id


def fill_segment_ids(row, provider_name, segments):
    row['segment_ids'] = []
    for segment in row['segments']:
        if ':' in segment:
            segmentation_name, segment_name = segment.split(':')
        else:
            segmentation_name, segment_name = None, segment

        segment_variants = segments.get(segment_name) or []
        if not segment_variants:
            available_segments = []
            for name, data in segments.items():
                segment_providers = {data_line['provider_name'] for data_line in data}
                if provider_name in segment_providers:
                    available_segments.append(name)
            exception_rows = [
                f'\nУ провайдера {provider_name}, не существует сегмента с именем {segment_name}\n'
                'доступные имена сегментов:\n'
            ]
            for name in available_segments:
                exception_rows.append(f'{name}\n')
            raise Exception(''.join(exception_rows))
        segment_id = None
        if len(segment_variants) == 1:
            segment_id = segment_variants[0]['id']
        if len(segment_variants) > 1:
            for segment_variant in segment_variants:
                if segmentation_name == segment_variant['segmentation_name']:
                    segment_id = segment_variant['id']
                    break

            if not segment_id:
                exception_rows = [
                    f'\nУ провайдера {provider_name}, существует несколько типов сегментаций для {segment_name}\n',
                    'нужно указать сегмент с явным указанием типа сегментации, возможные варианты:\n'
                ]
                for variant in segment_variants:
                    exception_rows.append(f'{variant["segmentation_name"]}:{segment_name}\n')
                raise Exception(''.join(exception_rows))

        row['segment_ids'].append(segment_id)


def fill_resource_type_id(row, resource_types):
    provider_name = row['provider']
    resource = row['resource']
    if resource not in resource_types[provider_name]:
        exception_rows = [
            f'\nУ провайдера {provider_name}, нет типа ресурса с именем {resource}\n',
            'список доступных типов ресурсов:\n',
        ]
        exception_rows.extend(sorted('{}\n'.format(res_type) for res_type in resource_types[provider_name]))
        raise Exception(''.join(exception_rows))
    row['resource_type_id'] = resource_types[provider_name][resource]


def fill_resource_id(row, d_client, unit_names, unit_ids):
    provider_name = row['provider']
    provider_id = row['provider_id']
    resource = row['resource']
    amount = row['amount']
    d_resources = d_client.provider_resources_by_segments(provider_id, row['resource_type_id'], row['segment_ids'])
    if not d_resources:
        exception_rows = [
            f'\nУ провайдера {provider_name}, не существует ресурса со следующими параметрами:\n',
            f'тип ресурса:\n    {resource}\n',
            'сегменты:\n',
        ]
        exception_rows.extend([f'    {segment}\n' for segment in row['segments']])
        raise Exception(''.join(exception_rows))
    if len(d_resources) > 1:
        exception_rows = [
            f'\nУ провайдера {provider_name}, существует несколько ресурсов с перечисленными типом и сегментацией\n',
            'Нужно указать дополнительные сегменты',
            f'тип ресурса:\n    {resource}\n',
            'сегменты:\n',
        ]
        exception_rows.extend([f'    {segment}\n' for segment in row['segments']])
        exception_rows.append('существующие ресурсы:\n')
        exception_rows.extend([f'    {res["key"]}\n' for res in d_resources])
        raise Exception(''.join(exception_rows))
    d_res = d_resources[0]
    row['resource_id'] = d_res['id']
    row['ensemble_id'] = d_res['unitsEnsembleId']
    unit = row['unit']
    if 'allowedUnitKeys' in d_res:
        allowed_units = d_res['allowedUnitKeys']
    else:
        allowed_units = [unit_ids[unit_id] for unit_id in d_res['allowedUnitIds']]
    row['allowed_units'] = allowed_units
    if amount != 'all' and unit not in unit_names:
        exception_rows = [
            f'\nДля указанного ресурса {resource}, не существует единицы измерения: {unit}\n',
            'Допустимые единицы измерения:\n',
        ]
        exception_rows.extend([f'   {unit_key}\n' for unit_key in sorted(allowed_units)])
        raise Exception(''.join(exception_rows))


def fill_transfer_data_ids(d_client, simple_transfers):

    unit_names, unit_ids, unit_ensemblies = get_all_units(d_client)
    all_d_providers = all_providers(d_client)
    providers, services = unique_services_and_providers(all_d_providers, simple_transfers)
    service_folders = get_service_folders(d_client, services)

    resource_types = get_resource_types(d_client, providers, all_d_providers)
    segments = get_segments_by_keys(d_client, providers, all_d_providers)

    for row in simple_transfers:
        provider_name = row['provider']
        provider_id = all_d_providers[provider_name]['id']
        row['provider_id'] = provider_id
        source_is_provider = 'source_service' not in row
        target_is_provider = 'target_service' not in row
        amount = row['amount']
        unit = row['unit']

        fill_segment_ids(row, provider_name, segments)
        fill_resource_type_id(row, resource_types)
        fill_resource_id(row, d_client, unit_names, unit_ids)

        ensemble_id = row['ensemble_id']
        unit_multipliers = get_unit_multipliers(unit_ensemblies, ensemble_id, available_units=row['allowed_units'])
        if amount != 'all':
            row['base_amount'] = amount * unit_multipliers[unit][0]
            row['base_unit'] = unit_multipliers[unit][1]

        if source_is_provider:
            row['source_service'] = all_d_providers[provider_name]['abc_service_id']
        if target_is_provider:
            row['target_service'] = all_d_providers[provider_name]['abc_service_id']

        source_folder_name = row.get('source_folder') or 'reserve' if source_is_provider else 'default'
        target_folder_name = row.get('target_folder') or 'reserve' if target_is_provider else 'default'
        row['source_folder_name'] = source_folder_name
        row['target_folder_name'] = target_folder_name
        row['source_folder_id'] = service_folders[row['source_service']][source_folder_name]['id']
        row['target_folder_id'] = service_folders[row['target_service']][target_folder_name]['id']

    return unit_ensemblies


def create_transfer_request(d_client, source_folder_id, target_folder_id, simple_transfers,
                            comment='Quota migration by cli tool'):
    def transfer_string(provider_id, resource_id, delta, units):
        return {
            'providerId': provider_id,
            'resourceId': resource_id,
            'delta': delta,
            'deltaUnitKey': units,
        }

    source_folder_quota_response = d_client.get_folder_quota(source_folder_id)
    source_folder_quota = {f['resourceId']: (f['quota'], f['quotaUnitKey']) for f in source_folder_quota_response}

    resource_transfers = defaultdict(list)
    for transfer_row in simple_transfers:
        provider_id = transfer_row['provider_id']
        res_id = transfer_row['resource_id']
        amount = transfer_row['amount']
        unit = transfer_row['unit']
        if amount == 'all':
            amount, unit = source_folder_quota[res_id]
        resource_transfers[source_folder_id].append(transfer_string(provider_id, res_id, -amount, unit))
        resource_transfers[target_folder_id].append(transfer_string(provider_id, res_id, amount, unit))

    quota_transfers = []
    for folder_id, res_transfers in resource_transfers.items():
        quota_transfers.append({
            'folderId': folder_id,
            'resourceTransfers': res_transfers,
        })
    transfer_request = {
        'requestType': 'QUOTA_TRANSFER',
        'description': comment,
        'addConfirmation': True,
        'parameters': {
            'quotaTransfers': quota_transfers
        }
    }
    return transfer_request


def create_provision(d_client, unit_ensemblies, provider_id, folder_id, account_id, simple_transfers, action='rise'):
    multiplier = 1 if action == 'drop' else -1
    if account_id is None:
        account_data = d_client.get_folder_provider_accounts(folder_id, provider_id)
        account_id = account_data[0]['id']

    current_provision = d_client.get_provisions(folder_id, account_id)
    current_provision = {f['resourceId']: (f['provided'], f['providedUnitKey']) for f in current_provision}
    updated_provision = []
    for transfer in simple_transfers:
        res_id = transfer['resource_id']
        resource_name = transfer['resource']
        ensemble_id = transfer['ensemble_id']
        current_amount, current_unit = current_provision.get(res_id, (0, ''))
        if current_amount:
            unit_multipliers = get_unit_multipliers(unit_ensemblies, ensemble_id,
                                                    available_units=transfer['allowed_units'])
            current_amount *= unit_multipliers[current_unit][0]
            current_unit = unit_multipliers[current_unit][1]
        transfer_amount = current_amount if transfer['amount'] == 'all' else transfer['base_amount']
        if transfer['amount'] == 'all':
            transfer['unit'] = current_unit
            transfer['base_unit'] = current_unit
        unit = transfer['base_unit']
        delta = transfer_amount * multiplier
        if delta == 0:
            continue
        new_amount = current_amount + delta
        if new_amount < 0 and action == 'rise':
            exception_rows = [
                f'\nНевозможно поднять требуемую квоту: {resource_name}\n',
                f'Спущенно сейчас: {current_amount} {unit}\n',
                f'Хотим поднять: {-delta} {unit}\n',
            ]
            raise Exception(''.join(exception_rows))
        updated_provision.append(
            {
                'providerId': provider_id,
                'resourceId': res_id,
                'provided': new_amount,
                'providedUnitKey': unit,
            }
        )
    return {
        'updatedProvisions': updated_provision
    }


def operation_description(line):
    return {
        'provider': '{}:{}'.format(line['provider'], line['segment']),
        'res': '{}:{}:{}'.format(line['resource'], line['amount'], line['units']),
        'to_service': line['target_service'],
        'from_service': line['source_service'],
    }


def provision_data(provider_id, res_id, amount, units):
    return {
        "updatedProvisions": [
            {
                "providerId": provider_id,
                "resourceId": res_id,
                "provided": amount,
                "providedUnitKey": units,
            }
        ]
    }
