import argparse
import datetime
import time
import copy

import yt.wrapper as yt
from infra.capacity_planning.abc_request_list.dispenser_requests.lib.dispenser import DispenserApi


CLUSTER = 'hahn'
DC = ('SAS', 'VLA', 'IVA', 'MAN', 'MYT')


def get_dispenser_data(dispenser_api, campaigns, clouds):
    path = '/quota-requests'
    params = {
        'lang': 'ru',
        'campaignOrder': campaigns if campaigns else [],
        'status': 'CONFIRMED',
    }

    timestamp = int(time.mktime(time.strptime(time.strftime('%Y %m %d %H %M'), '%Y %m %d %H %M')))
    data = dispenser_api.get_request(path=path, params=params).json()
    result = []
    for issue in data['result']:
        schema = {
            'timestamp': timestamp,
            'order_id': issue['id'],
            'ticket': issue['trackerIssueKey'],
            'abc_id': issue['project']['abcServiceId']
        }
        for resource in issue["changes"]:
            cloud = resource['service']['key']
            if cloud in clouds:
                base_row = copy.deepcopy(schema)
                base_row['cloud'] = cloud
                base_row['campaign_order_id'] = resource['order']['id']
                order_date = datetime.datetime.strptime(resource['order']['orderDate'], '%Y-%m-%d')
                base_row['order_date'] = int(order_date.timestamp())
                segment_keys = resource['segmentKeys']
                if cloud in ('yp', 'gencfg'):
                    base_row['dc'] = segment_keys[0]
                    if len(segment_keys) == 2:
                        base_row['dc'] = segment_keys[0] if segment_keys[0] in DC else segment_keys[1]
                        base_row['segment'] = segment_keys[0] if segment_keys[0] not in DC else segment_keys[1]
                else:
                    base_row['segment'] = segment_keys[0]
                resource_name = resource["resource"]['key'].removesuffix('_segmented')
                base_row['resource'] = resource_name

                for state, postfix in (('total', ''), ('ready', 'Ready'), ('allocated', 'Allocated'), ('remaining', '')):
                    row = copy.deepcopy(base_row)
                    row['state'] = state
                    row['unit'] = resource['amount']['unit']
                    if state != 'remaining':
                        row['value'] = resource[f'amount{postfix}']['value']
                    else:
                        row['value'] = resource['amount']['value'] - resource['amountAllocated']['value']
                    result.append(row)
    return result


def args_parse():
    parser = argparse.ArgumentParser()
    parser.add_argument('--token', default=None, required=True)
    parser.add_argument('--campaigns', nargs='+', type=int, required=True)
    parser.add_argument('--clouds', nargs='+', type=str, required=True)
    args = parser.parse_args()
    token = args.token
    campaigns = args.campaigns
    clouds = list(map(str.lower, args.clouds))
    return token, campaigns, clouds


def main():
    token, campaigns, clouds = args_parse()

    dispenser_api = DispenserApi(dispenser_token=token)
    data = get_dispenser_data(dispenser_api, campaigns, clouds)

    schema = [
        {'name': 'timestamp', 'type': 'datetime'},
        {'name': 'order_id', 'type': 'int64'},
        {'name': 'ticket', 'type': 'string'},
        {'name': 'abc_id', 'type': 'int64'},
        {'name': 'campaign_order_id', 'type': 'int64'},
        {'name': 'cloud', 'type': 'string'},
        {'name': 'dc', 'type': 'string'},
        {'name': 'segment', 'type': 'string'},
        {'name': 'order_date', 'type': 'datetime'},
        {'name': 'resource', 'type': 'string'},
        {'name': 'state', 'type': 'string'},
        {'name': 'value', 'type': 'int64'},
        {'name': 'unit', 'type': 'string'},
    ]
    yt_client = yt.YtClient(CLUSTER, token=token)
    path_default = '<append=true>//home/capacity_planning/reserves/d_quota_requests/quota_requests'
    path_daily = '//home/capacity_planning/reserves/d_quota_requests/quota_requests-daily'
    yt_client.create("table", path=path_default, attributes={"schema": schema}, ignore_existing=True)
    yt_client.write_table(path_default, data)
    yt_client.create("table", path=path_daily, attributes={"schema": schema}, force=True)
    yt_client.write_table(path_daily, data)


if __name__ == '__main__':
    main()
