import re
import copy
import argparse
import datetime
from collections import defaultdict

import yt.wrapper as yt
from startrek_client import Startrek
from infra.capacity_planning.library.python.abc_tree import AbcApi

STARTREK_API_BASE_URL = 'https://st-api.yandex-team.ru/'


def parse_st_tickets(token, head_ticket_slug, abc_api, month=None):
    client = Startrek(useragent='python', token=token, base_url=STARTREK_API_BASE_URL)
    head_ticket = client.issues[head_ticket_slug]

    reg_exp = re.compile(r'\*\*(?P<key>.*?)\*\*\n%%\n(?P<value>.*?)\n%%\n', re.DOTALL)

    columns = {('date', 'datetime'), }
    total_data = []
    change_data = defaultdict(lambda: defaultdict(list))
    for link in head_ticket.links:
        if link.type.id != 'subtask':
            continue
        sub_task_description = link.object.description

        values = reg_exp.findall(sub_task_description)
        row = {key: val for key, val in values}
        row['ticket'] = link.object.key
        row['status'] = 'closed' if link.object.status.key == 'closed' else 'open'

        abc_service = link.object.abcService[0] if link.object.abcService else None
        row['ABC сервис'] = abc_service.name if abc_service else None
        row['VS'] = get_vs(int(abc_service.id), abc_api) if abc_service else None

        columns.update([(row_key, 'string') for row_key in row])

        if 'Ресурсы' in row:
            new_split_rows, res_columns = split_resources(change_data, row, row['Ресурсы'], month)
            total_data.extend(new_split_rows)
            columns.update(res_columns)
        else:
            total_data.append(row)

    return total_data, columns, change_data


def get_string_date(string_date, ticket):
    if string_date == 'Май':
        month = 5
    elif string_date == 'Июнь':
        month = 6
    elif string_date == 'Июль':
        month = 7
    elif string_date == 'Август':
        month = 8
    elif string_date == 'Сентябрь':
        month = 9
    elif string_date == 'Октябрь':
        month = 10
    elif string_date == 'Ноябрь':
        month = 11
    elif string_date == 'Декабрь':
        month = 12
    else:
        raise Exception(f'month error in ticket {ticket}')
    datetime_date = datetime.datetime(2022, month, 1)
    datetime_int = int(datetime_date.strftime("%s"))

    return datetime_int


def get_vs(abc_id, abc_api):
    slug = abc_api.get_service_slug(abc_id)
    if 'vs' in abc_api.service_tree[abc_id]['tags']:
        return abc_api.get_service_name(slug)
    else:
        return abc_api.get_service_name(abc_api.get_service_vs_or_top(slug, abc_id))


def split_resources(change_data, initial_row, res_text, month=None):
    new_rows = []
    res_columns = set()
    ticket = initial_row['ticket']
    name_keys = ('Провайдер', 'Датацентр', 'Кластер', 'Сегментация', 'Когда будут нужны ресурсы')
    for row in res_text.split('Ресурсы\n'):
        row = row.strip()
        if not row:
            continue
        new_row = copy.deepcopy(initial_row)
        res_dict = {}
        for res_part in row.split('\n'):
            res_key, res_val = res_part.split(' - ')
            if res_key in name_keys:
                res_dict[res_key] = res_val
                res_columns.add((res_key, 'string'))
            else:
                try:
                    res_dict[res_key] = float(res_val.replace(',', '.'))
                except ValueError:
                    raise Exception(f'value error in ticket {ticket}, value = {res_val}')
                res_columns.add((res_key, 'double'))

        provider = res_dict.get('Провайдер') or ''
        raw_segment = res_dict.get('Датацентр') or res_dict.get('Кластер') or res_dict.get('Сегментация') or '-'
        abc_name = initial_row['ABC сервис']
        vs_name = initial_row['VS']
        status = initial_row['status']
        segments = raw_segment.split(', ')
        if month:
            datetime_date = datetime.datetime(2022, month, 1)
            date = int(datetime_date.strftime("%s"))
        else:
            if 'Когда будут нужны ресурсы' not in res_dict:
                raise Exception(f'key error in resource dates in ticket {ticket}')
            string_month = res_dict['Когда будут нужны ресурсы']
            date = get_string_date(string_month, ticket)

        for segment in segments:
            for res_key, res_val in res_dict.items():
                if res_key in name_keys:
                    continue
                update_collect_data(change_data, (provider, segment, abc_name, vs_name, ticket, status), res_key, date, res_val)

        new_row.update(res_dict)
        new_rows.append(new_row)

    return new_rows if new_rows else [initial_row], res_columns


def update_collect_data(collector_data, prov_segm__key, res_key, date, val):
    res_list = collector_data[prov_segm__key][res_key]
    if not res_list:
        res_list.extend([
            [int(datetime.datetime(2022, 5, 1).strftime("%s")), 0.0, 0.0],
            [int(datetime.datetime(2022, 6, 1).strftime("%s")), 0.0, 0.0],
            [int(datetime.datetime(2022, 7, 1).strftime("%s")), 0.0, 0.0],
            [int(datetime.datetime(2022, 8, 1).strftime("%s")), 0.0, 0.0],
            [int(datetime.datetime(2022, 9, 1).strftime("%s")), 0.0, 0.0],
            [int(datetime.datetime(2022, 10, 1).strftime("%s")), 0.0, 0.0],
            [int(datetime.datetime(2022, 11, 1).strftime("%s")), 0.0, 0.0],
            [int(datetime.datetime(2022, 12, 1).strftime("%s")), 0.0, 0.0],
            [int(datetime.datetime(2023, 1, 1).strftime("%s")), 0.0, 0.0],
        ])
    insert_index = len(res_list)
    _, _, prev_finish = res_list[-1] if res_list else [None, 0.0, 0.0]

    for i, cur_row in reversed(list(enumerate(res_list))):
        cur_date, _, _ = cur_row
        if date < cur_date:
            cur_row[1] += val
            cur_row[2] += val
            insert_index = i
        elif date == cur_date:
            cur_row[2] += val
            insert_index = None
        else:
            break
        _, _, prev_finish = res_list[i - 1] if i > 0 else [None, 0.0, 0.0]

    if insert_index is not None:
        res_list.insert(insert_index, [date, prev_finish, prev_finish + val])


def update_yt_data(token, path, schema, data):
    cluster = 'hahn'
    yt_client = yt.YtClient(cluster, token=token)

    path_parts = path.split('/')
    folder_path = '/'.join(path_parts[:-1])
    table_name = path_parts[-1]
    if table_name in yt_client.list(folder_path):
        yt_client.remove(path)
    yt_client.create("table", path,
                     ignore_existing=True,
                     attributes={
                         "schema": schema
                     }
                     )
    yt_client.write_table(path, data)


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--token', required=True)
    parser.add_argument('--main-ticket', required=True)
    parser.add_argument('--month', type=int)
    return parser.parse_args()


def main():
    args = parse_args()
    token = args.token
    main_ticket = args.main_ticket
    month = args.month

    abc_api = AbcApi(token=token)
    abc_api.load_abc_service_tree()

    total_data, columns, change_data = parse_st_tickets(token, main_ticket, abc_api, month)

    total_schema = []
    for col_name, col_type in columns:
        total_schema.append(
            {
                'name': col_name,
                'type': col_type,
            }
        )
    path = f'//home/capacity_planning/recomposing/{main_ticket}'
    update_yt_data(token, path, total_schema, total_data)

    change_rows = []
    for provider_segment, provider_segment_data in change_data.items():
        provider, segment, abc_name, vs_name, ticket, status = provider_segment
        for res_name, value_data in provider_segment_data.items():
            for date, start, finish in value_data:
                start_row = {
                    'ticket': ticket,
                    'status': status,
                    'ABC': abc_name,
                    'VS': vs_name,
                    'provider': provider,
                    'segment': segment,
                    'res_name': res_name,
                    'date': date - 1,
                    'value': start,
                }
                finish_row = {
                    'ticket': ticket,
                    'status': status,
                    'ABC': abc_name,
                    'VS': vs_name,
                    'provider': provider,
                    'segment': segment,
                    'res_name': res_name,
                    'date': date,
                    'value': finish,
                }
                change_rows.append(start_row)
                change_rows.append(finish_row)
    change_schema = [
        {'name': 'ticket', 'type': 'string'},
        {'name': 'status', 'type': 'string'},
        {'name': 'ABC', 'type': 'string'},
        {'name': 'VS', 'type': 'string'},
        {'name': 'provider', 'type': 'string'},
        {'name': 'segment', 'type': 'string'},
        {'name': 'res_name', 'type': 'string'},
        {'name': 'date', 'type': 'datetime'},
        {'name': 'value', 'type': 'double'},
    ]
    path = f'//home/capacity_planning/recomposing/{main_ticket}-changes'
    update_yt_data(token, path, change_schema, change_rows)
