"""
Collects data from YP and adds them to the YT table.
"""

import yt.wrapper as yt
import json
from collections import OrderedDict
from datetime import date
import time
import subprocess

CLUSTER = 'hahn'
YT_PATH = '<append=true>//home/capacity_planning/reserves/capacity/yp/yp_add'
yt_client = yt.YtClient(CLUSTER)


def create_table():
    """
    Creating table for YT.
    :return: None
    """
    yt_client.create("table", YT_PATH,
                     ignore_existing=True,
                     attributes={
                         "schema":
                             [
                                 {"name": "Data", "type": "datetime"},
                                 {"name": "dc", "type": "string"},
                                 {"name": "segment", "type": "string"},
                                 {"name": "cpu", "type": "int64"},
                                 {"name": "memory", "type": "int64"},
                                 {"name": "hdd_capacity", "type": "int64"},
                                 {"name": "hdd_bandwidth", "type": "int64"},
                                 {"name": "ssd_capacity", "type": "int64"},
                                 {"name": "ssd_bandwidth", "type": "int64"},
                                 {"name": "network_bandwidth", "type": "int64"},
                                 {"name": "gpu_tesla_v100", "type": "int64"},
                                 {"name": "gpu_tesla_v100_nvlink", "type": "int64"},
                                 {"name": "gpu_geforce_1080ti", "type": "int64"},
                                 {"name": "gpu_tesla_k40", "type": "int64"},
                                 {"name": "gpu_tesla_m40", "type": "int64"},
                                 {"name": "gpu_tesla_a100", "type": "int64"},
                                 {"name": "gpu_tesla_a100_80g", "type": "int64"},
                             ]
                     }
                     )


def get_data(reserves):
    """
    collect data from source
    :return: data in format: list cosisting of Orderdict
    """
    datatime = time.mktime(time.strptime(str(date.today()), "%Y-%m-%d"))
    result = []

    for element in reserves:

        row = OrderedDict()
        dc = element[0]
        segment = element[1]
        data = element[2][0]
        row["Data"] = int(datatime)
        row["dc"] = dc
        row["segment"] = segment

        row["cpu"] = data["cpu"]["capacity"] // 1000
        row["memory"] = data["memory"]["capacity"]
        try:
            row["hdd_capacity"] = data["disk_per_storage_class"]["hdd"]["capacity"]
            row["hdd_bandwidth"] = data["disk_per_storage_class"]["hdd"]["bandwidth"]
            row["ssd_capacity"] = data["disk_per_storage_class"]["ssd"]["capacity"]
            row["ssd_bandwidth"] = data["disk_per_storage_class"]["ssd"]["bandwidth"]
        except:
            pass
        row["network_bandwidth"] = data["network"]["bandwidth"]
        result.append(row)

    return result


def main():
    DC_LIST = ["sas", "man", "vla", "iva", "myt"]

    NODE_SEGMENT = ['dev', 'default', 'gpu-dev', 'gpu-default', 'sandbox', 'dbaas', 'saas', 'distbuild', 'yabs']
    arg = []
    for dc in DC_LIST:
        for segment in NODE_SEGMENT:
            if dc in ('myt', 'iva') and segment in ('saas', 'distbuild'):
                continue
            command = "ya tool yp get node_segment {} --selector /status/schedulable_resources --format json --address {}".format(
                segment, dc)
            data = json.loads(subprocess.check_output(command, shell=True))
            arg.append([dc, segment, data])

    create_table()
    result = get_data(arg)
    yt_client.write_table(YT_PATH, result)


if __name__ == '__main__':
    main()
