"""
Collect data of qloud capacity
"""

import requests
from requests.exceptions import ConnectionError, ChunkedEncodingError
import time
from collections import OrderedDict, defaultdict
import argparse
from datetime import date
import yt.wrapper as yt
from infra.capacity_planning.library.python.base import retry

CLUSTER = 'hahn'
YT_PATH = '<append=true>//home/capacity_planning/reserves/capacity/qloud/qloud'
yt_client = yt.YtClient(CLUSTER)


def create_table():
    """
    Create table in YT
    :return: None
    """
    yt_client.create("table", YT_PATH,
                     ignore_existing=True,
                     attributes={
                         "schema":
                             [
                                 {"name": "data", "type": "datetime"},
                                 {"name": "installation", "type": "string"},
                                 {"name": "dc", "type": "string"},
                                 {"name": "segment", "type": "string"},
                                 {"name": "cpu", "type": "double"},
                                 {"name": "memory", "type": "int64"},
                             ]
                     }
                     )


QLOUD_DOMAINS = ["qloud.yandex-team.ru", "qloud-ext.yandex-team.ru"]


@retry(exceptions=(ConnectionError, ChunkedEncodingError))
def collect_projects(domain, headers):
    """
    Collect capacity per segment
    :param domain: name of domain
    :param headers: OAuth token
    :return: capacity data
    """
    hosts_data = requests.get(
        "https://{}/api/v1/hosts/search".format(domain),
        headers=headers
    ).json()

    datatime = time.mktime(time.strptime(str(date.today()), "%Y-%m-%d"))

    hosts_stat = defaultdict(lambda: defaultdict(lambda: defaultdict(int)))

    for host in hosts_data:
        cpu = host["cpu"]
        mem = host["memoryBytes"]
        dc = host.get("dataCenter", "unknown")
        segment = host.get("segment", "unknow")

        hosts_stat[dc][segment]["cpu"] += cpu
        hosts_stat[dc][segment]["memory"] += mem

    capacity = []
    for dc, dc_data in hosts_stat.items():
        for segment, segment_data in dc_data.items():
            row = OrderedDict()

            row["installation"] = domain.split(".")[0]
            row["data"] = int(datatime)
            row["dc"] = dc
            row["segment"] = segment
            row.update(segment_data)

            capacity.append(row)

    return capacity


def get_full_qloud_domain(domain):
    """
    Convert domain name to full url
    :param domain: domain name
    :return: full domain name
    """
    if "." in domain:
        assert domain in QLOUD_DOMAINS
        return domain

    for el in QLOUD_DOMAINS:
        if el.split(".")[0] == domain:
            return el

    assert False


def args_parse():
    parser = argparse.ArgumentParser()
    parser.add_argument('--secret_token_YP', default=None)
    parser.add_argument('--secret_token_ABC', default=None)
    args = parser.parse_args()
    return args.secret_token_ABC


def main():
    abc_token = args_parse()
    create_table()
    headers = {'Authorization': 'OAuth {}'.format(abc_token)}

    data = []
    for domain in QLOUD_DOMAINS:
        domain = get_full_qloud_domain(domain)
        data += collect_projects(domain, headers)
    yt_client.write_table(YT_PATH, data)
