#!/usr/bin/env python
# coding=utf-8
"""
Collects data on all active issued credits and adds them to the YT table.
"""
import copy
import datetime

from startrek_client import Startrek
import yt.wrapper as yt
from collections import OrderedDict
import time
import argparse

CLUSTER = 'hahn'
YT_PATH = '<append=true>//home/capacity_planning/reserves/credits/credits'
YT_PATH_DYNAMIC = '//home/capacity_planning/reserves/credits/credits-dynamic'

DC_LIST = ["sas", "man", "vla", "iva", "myt"]


def get_task_credit(task):
    """
    Getting credits from tickets
    :param task: issue from what credits are got
    :return: dictionary consisting of credit iformation: dc, cpu
    """
    credit_line_prefix = u"кредит "

    res = []

    for comment in task.comments:

        for full_line in comment.text.split("\n"):
            line = full_line.strip().lower().replace("==", "=")

            if not line.startswith(credit_line_prefix):
                continue

            line = line.split()
            if len(line) < 4:
                continue

            assert line[0] == credit_line_prefix.strip()
            dc = line[1]
            credit_size = line[2]
            resource_name = line[3]

            if dc not in DC_LIST:
                continue

            if resource_name not in [u"core", u"cores", u"ядро", u"ядра", u"ядер"]:
                continue

            try:
                credit_size = float(credit_size)
            except:
                continue

            resource = OrderedDict()
            resource["resource"] = "cores"
            resource["cloud"] = "credits"
            resource["dc"] = dc
            resource["cores"] = credit_size
            res.append(resource)
    return res


def get_issue(client):
    """
    Getting issues with 'rtc_credit' tag and parsing parameters
    :param client: Startrek client
    :return: parsed data
    """
    data = []
    issues = client.issues.find('tags: rtc_credit')
    datatime = time.mktime(time.strptime(str(datetime.date.today()), "%Y-%m-%d"))
    for issue in issues:
        dc_cerdit = get_task_credit(issue)
        for credits in dc_cerdit:
            if issue.status.key != "closed":
                row = OrderedDict()
                row["date"] = int(datatime)
                row["author"] = issue.createdBy.id
                row["ticket"] = issue.key

                if issue.start:
                    row["start_date"] = int(time.mktime(time.strptime(issue.start, "%Y-%m-%d")))
                if issue.end:
                    row["end_date"] = int(time.mktime(time.strptime(issue.end, "%Y-%m-%d")))
                if issue.deadline:
                    row["deadline"] = int(time.mktime(time.strptime(issue.deadline, "%Y-%m-%d")))
                row["resources_cores"] = credits["cores"]
                row["dc"] = credits["dc"]
                row["segment"] = "default"
                if issue.abcService:
                    row["ABC_id"] = int(issue.abcService[0].id)
                    row["ABC_name"] = issue.abcService[0].name
                tags = issue.tags
                if "ftq" in tags and issue.queue.key == "FTQ":
                    row["tag"] = "free_tier"

                if "rtc_credit_gencfg" in tags:
                    row["provider"] = "gencfg"
                elif "rtc_credit_qloud" in tags:
                    row["provider"] = "qloud"
                else:
                    row["provider"] = "yp"
                data.append(row)
    return data


def get_dynamic_data(initial_data):
    dynamic_data = []
    base_data = []
    for row in initial_data:
        if row['resources_cores'] == 0:
            continue
        row['type'] = 'credit' if row['resources_cores'] > 0 else 'debt'
        if not row.get('start_date'):
            row['start_date'] = row['date']
        base_data.append(row)
    start = base_data[0]['date']
    finish = start + int(datetime.timedelta(days=365).total_seconds())
    for date in range(start, finish, int(datetime.timedelta(days=1).total_seconds())):
        for row in base_data:
            row['date'] = date
            if row['resources_cores'] and row.get('deadline') and start < row['deadline'] <= date:
                row['resources_cores'] = 0.0
        dynamic_data.extend(copy.deepcopy(base_data))
    return dynamic_data


def args_parse():
    parser = argparse.ArgumentParser()
    parser.add_argument('--secret_token', default=None)
    args = parser.parse_args()
    return args.secret_token


def main():
    token = args_parse()
    yt_client = yt.YtClient(CLUSTER, token=token)
    schema_default = [
        {"name": "date", "type": "datetime"},
        {"name": "author", "type": "string"},
        {"name": "ticket", "type": "string"},
        {"name": "start_date", "type": "datetime"},
        {"name": "end_date", "type": "datetime"},
        {"name": "deadline", "type": "datetime"},
        {"name": "resources_cores", "type": "double"},
        {"name": "dc", "type": "string"},
        {"name": "segment", "type": "string"},
        {"name": "ABC_id", "type": "int64"},
        {"name": "ABC_name", "type": "string"},
        {"name": "tag", "type": "string"},
        {"name": "provider", "type": "string"},
    ]
    schema_dynamic = copy.deepcopy(schema_default)
    schema_dynamic.append({"name": "type", "type": "string"})

    st_client = Startrek(token=token, useragent="python")
    data = get_issue(st_client)
    yt_client.create("table", YT_PATH, attributes={"schema": schema_default}, ignore_existing=True)
    yt_client.write_table(YT_PATH, data)

    dynamic_data = get_dynamic_data(initial_data=data)
    yt_client.create("table", YT_PATH_DYNAMIC, attributes={"schema": schema_dynamic}, force=True)
    yt_client.write_table(YT_PATH_DYNAMIC, dynamic_data)


if __name__ == '__main__':
    main()
