"""
Collect information about gencfg migrations
"""

import requests
import time
from collections import OrderedDict
from datetime import date

import yt.wrapper as yt
from infra.capacity_planning.library.python.base import retry

CLUSTER = 'hahn'
YT_PATH = '<append=true>//home/capacity_planning/reserves/migrations/gencfg/gencfg'
yt_client = yt.YtClient(CLUSTER)


def create_table():
    """
    Creating table in YT
    :return: None
    """
    yt_client.create("table", YT_PATH,
                     ignore_existing=True,
                     attributes={
                         "schema":
                             [
                                 {"name": "data", "type": "datetime"},
                                 {"name": "abc", "type": "int64"},
                                 {"name": "gencfg_groups", "type": "string"},
                                 {"name": "dc", "type": "string"},
                                 {"name": "cpu", "type": "double"},
                                 {"name": "hdd", "type": "double"},
                                 {"name": "ipv4", "type": "double"},
                                 {"name": "memory", "type": "double"},
                                 {"name": "ssd", "type": "double"},
                                 {"name": "ticket", "type": "string"},
                             ]
                     }
                     )


@retry
def get_data():
    data = requests.get('https://yp-quota-distributor.n.yandex-team.ru/gencfg/stats', verify=False)
    return data.json()


def parse_data(data):
    """
    Format data from migrator
    :return: formatted data
    """
    result = []
    datatime = time.mktime(time.strptime(str(date.today()), "%Y-%m-%d"))
    for line in data["openMigrations"]:
        gencfg_groups = set(elem.lower() for elem in line["gencfg_groups"])
        for dc, qouta in line["quota"].items():
            row = OrderedDict()
            row["data"] = int(datatime)
            row["abc"] = line["abc"]

            if dc in gencfg_groups:
                row["gencfg_groups"] = dc
            row["dc"] = dc
            row["cpu"] = qouta["cpu"]
            row["hdd"] = qouta["hdd"]
            row["ipv4"] = float(qouta["ipv4"])
            row["memory"] = qouta["memory"]
            row["ssd"] = qouta["ssd"]
            row["ticket"] = line["ticket"].split('/')[-1]
            result.append(row)
    return result


def main():
    create_table()
    data = get_data()
    result = parse_data(data)
    yt_client.write_table(YT_PATH, result)
