import requests
from requests.exceptions import ConnectionError, ChunkedEncodingError

from infra.capacity_planning.library.python import abc_tree
from infra.capacity_planning.library.python.base import retry

import time
from collections import OrderedDict
import argparse
from datetime import date

import yt.wrapper as yt

CLUSTER = 'hahn'
YT_PATH = '<append=true>//home/capacity_planning/reserves/quotas/qloud/qloud'
yt_client = yt.YtClient(CLUSTER)

QLOUD_DOMAINS = ["qloud.yandex-team.ru", "qloud-ext.yandex-team.ru"]


def create_table():
    """
    Creating table in YT
    :param client: YT client
    :param path: path to table
    :return: None
    """
    yt_client.create("table", YT_PATH,
                     ignore_existing=True,
                     attributes={
                         "schema":
                             [
                                 {"name": "data", "type": "datetime"},
                                 {"name": "installation", "type": "string"},
                                 {"name": "project", "type": "string"},
                                 {"name": "segment", "type": "string"},
                                 {"name": "dc", "type": "string"},
                                 {"name": "cpu", "type": "double"},
                                 {"name": "memory", "type": "double"},
                             ]
                     }
                     )


def cloud(domain):
    """
    return cloud from URL
    :param domain: URL
    :return: cloud
    """
    return domain.split(".")[0]


def get_projects_to_abc(headers, domain, abc_api_holder):
    """
    Getting ABC id and projects
    :param headers: authorisation token
    :param domain: cloud domain (URL)
    :param abc_api_holder:information about abc services
    :return:
    """

    url = "https://{}/api/v1/project/".format(domain)

    resp = requests.get(url, headers=headers, verify=False)
    resp.raise_for_status()

    projects_to_abc = {}

    for project_data in resp.json():
        project_name = project_data["projectName"]
        if 'abcId' in project_data['metaInfo']:
            project_id = project_data['metaInfo']['abcId']
            project_abc = abc_api_holder.get_service_slug(project_id)
            if project_abc is None:
                project_abc = "abc.%s" % project_id
        else:
            project_abc = "abc_unknown.{}.{}".format(cloud(domain), project_name)

        projects_to_abc[project_name] = project_abc

    return projects_to_abc


def collect_projects(domain, project, project_data):
    """
    Format data from each project
    :param domain: cloud domain
    :param project: project name
    :param project_data: data from each project
    :return: formatted data
    """

    result = []
    datatime = time.mktime(time.strptime(str(date.today()), "%Y-%m-%d"))
    dc_s = set()
    for resource in project_data["quota"]:
        dc_s.add(resource["location"])
    for dc in dc_s:
        row = OrderedDict()
        row["data"] = int(datatime)
        row["project"] = project
        row["dc"] = dc
        row["installation"] = domain.split(".")[0]
        for resource in project_data["quota"]:
            if dc == resource["location"]:
                row["segment"] = resource["hardwareSegment"]
                if "memory" in resource["resource"].lower():
                    row["memory"] = resource["quota"] * (1024 * 1024 * 1024)
                elif "cpu" in resource["resource"].lower():
                    row["cpu"] = resource["quota"]
        result.append(row)
    return result


def get_full_qloud_domain(domain):
    """
    Getting domain from cloud
    :param domain: cloud
    :return: domain
    """
    if "." in domain:
        assert domain in QLOUD_DOMAINS
        return domain

    for el in QLOUD_DOMAINS:
        if el.split(".")[0] == domain:
            return el

    assert False


def get_project_data(self, project):
    """
    Collect resources from project
    :param self: ABC class
    :param project: project name
    :return: quotas and usage from project
    """
    if self._projects_data and project in self._projects_data:
        return self._projects_data[project]

    url = 'https://{}/api/v1/project/{}/quota'.format(self._domain, project)
    resp = requests.get(url, headers=self.headers, verify=False)
    resp.raise_for_status()

    project_data = resp.json()

    segments_resource_usage = self._get_usage(project_data)
    segments_resource_quota = self._get_quota(project_data)

    res = {
        "usage": segments_resource_usage,
        "quota": segments_resource_quota
    }

    return res


@retry(exceptions=(ConnectionError, ChunkedEncodingError, ConnectionResetError))
def get_data(token):
    abc_api_holder = abc_tree.load_abc_api(token=token, reload=False)
    headers = {'Authorization': 'OAuth {}'.format(token)}
    data = []

    for domain in QLOUD_DOMAINS:
        domain = get_full_qloud_domain(domain)
        projects = sorted(get_projects_to_abc(headers, domain, abc_api_holder).keys())

        for i, project in enumerate(projects):
            project_data = requests.get(
                'https://{}/api/v1/project/{}/quota'.format(domain, project),
                headers=headers
            ).json()
            data += collect_projects(domain, project, project_data)
    return data


def arg_parser():
    parser = argparse.ArgumentParser()
    parser.add_argument('--secret_token_YP', default=None)
    parser.add_argument('--secret_token_ABC', default=None)
    args = parser.parse_args()

    return args.secret_token_ABC


def main():
    token = arg_parser()
    create_table()
    data = get_data(token)

    yt_client.write_table(YT_PATH, data)
