'''
Collect data with strict sheme for datalens + union with date field in one table
'''

import yt.wrapper as yt
from collections import OrderedDict, defaultdict
from datetime import date
import time
import argparse
from requests.exceptions import ConnectionError, ChunkedEncodingError

from infra.capacity_planning.library.python.base_api import BaseApi
from infra.capacity_planning.library.python.yt_tables import create_table
from infra.capacity_planning.library.python.base import retry

cluster = 'hahn'
yt_client = yt.YtClient(cluster)
table_path = '<append=true>//home/capacity_planning/gpu_idle/abc_history_dataset_cost'


GPU_CARDS = ['gpu_tesla_v100', 'gpu_tesla_k40', 'gpu_tesla_m40', 'gpu_geforce_1080ti', 'gpu_tesla_a100_80g', 'gpu_tesla_a100']


@retry(exceptions=(ConnectionError, ChunkedEncodingError, ConnectionResetError))
def collect_service_names(abc_client, service_ids):
    # не params, потому что пейджинг работает выдавая слудеющий url, а не токен следующей страницы
    url = '/api/v4/services/?' + '&'.join([
        'id__in=' + ','.join(service_ids) if service_ids else '',
        'fields=id,name',
    ])
    service_names = abc_client._full_paged_data_by_url(url)
    return {str(row['id']): row['name'] for row in service_names}


def collect_qouta_free(abc_client, prices):
    daily_path = '//home/capacity_planning/gpu_idle/abc_day_cost'
    create_table(yt_client, table_path)

    data = defaultdict(lambda: defaultdict(float))
    today = date.today().strftime('%Y-%m-%d')
    datatime = time.mktime(time.strptime(today, '%Y-%m-%d'))

    table = yt_client.read_table('//home/data_com/cubes/yp/yp_accounts/{}'.format(today))

    rows = []
    service_ids = set()
    for row in table:
        if row['segment'] in ('gpu-default', 'gpu-dev') and 'abc' in row['account_id']:
            rows.append(row)
            account_id = row['account_id']
            service_id = account_id.split(':')[-1]
            service_ids.add(service_id)

    service_names = collect_service_names(abc_client, service_ids)

    for row in rows:
        account_id = row['account_id']
        service_id = account_id.split(':')[-1]
        segm = row['segment']
        dc = row['geo']
        name = '{}_{}_{}'.format(account_id, segm, dc)

        if row['quota_type'] in ('total_quota', 'abcd_quota', 'allocated_quota'):
            postfix = row['quota_type'].split('_')[-1]
            for gpu in GPU_CARDS:
                field_name = '{}_{}'.format(gpu, postfix)
                data[name][field_name] += row[gpu]

        if name in data:
            data[name]['account_id'] = service_names[service_id]['ru']
            data[name]['segment'] = segm
            data[name]['date'] = int(datatime)
            data[name]['dc'] = dc

    id_list = OrderedDict()
    for name, line in data.items():
        flag = 0
        for id, elem in line.items():
            if 'abcd' in id:
                flag = 1
                break
        if flag:
            id_list[line['account_id']] = 1
    result = []
    for name, line in data.items():
        for gpu in GPU_CARDS:
            row = OrderedDict()
            row['date'] = line['date']
            row['dc'] = line['dc']
            row['account_id'] = line['account_id']
            row['segment'] = line['segment']
            if id_list.get(line['account_id']) and line.get(gpu + '_allocated') is not None:
                if line.get(gpu + '_abcd') is not None:
                    row['gpu'] = int(line[gpu + '_abcd'] - line[gpu + '_allocated'])
                else:
                    row['gpu'] = int(line[gpu + '_total'] - line[gpu + '_allocated'])
            row['type_gpu'] = gpu
            row['type_quota'] = 'free'
            if row.get('gpu'):
                row['gpu_cost'] = row['gpu'] * prices[gpu]
                result.append(row)
    yt_client.write_table(daily_path, result)
    for elem in result:
        elem['gpu_type'] = elem['type_gpu']
        elem.pop('type_gpu')
        elem['quota_type'] = elem['type_quota']
        elem.pop('type_quota')
        elem['account_id'] += ' FREE'
    yt_client.write_table(table_path, result)


service_url = 'https://abc-back.yandex-team.ru/api/v4/services/?id__in={id}&fields=name'


def collect_allocated(abc_client, prices):
    daily_path = '<append=true>//home/capacity_planning/gpu_idle/abc_day_cost'
    create_table(yt_client, table_path)

    data = OrderedDict()
    today = date.today().strftime('%Y-%m-%d')
    datatime = time.mktime(time.strptime(today, '%Y-%m-%d'))

    table = yt_client.read_table('//home/data_com/cubes/yp/yp_accounts/{}'.format(today))
    rows = []
    service_ids = set()
    for row in table:
        if row['segment'] in ('gpu-default', 'gpu-dev') and 'abc' in row['account_id']:
            rows.append(row)
            account_id = row['account_id']
            service_id = account_id.split(':')[-1]
            service_ids.add(service_id)

    service_names = collect_service_names(abc_client, service_ids)

    for line in rows:
        account_id = line['account_id']
        segm = line['segment']
        dc = line['geo']

        name = '{}_{}_{}'.format(account_id, segm, dc)
        if line['quota_type'] == 'allocated_quota':
            for gpu in GPU_CARDS:
                data[name][gpu] += line[gpu]

        if name in data:
            data[name]['account_id'] = service_names[service_id]['ru']
            data[name]['segment'] = segm
            data[name]['date'] = int(datatime)
            data[name]['dc'] = dc

    result = []
    for name, line in data.items():
        for gpu in GPU_CARDS:
            if line.get(gpu):
                row = OrderedDict()
                row['date'] = line['date']
                row['dc'] = line['dc']
                row['account_id'] = line['account_id']
                row['segment'] = line['segment']
                row['gpu'] = line[gpu]
                row['gpu_cost'] = row['gpu'] * prices[gpu]
                row['type_gpu'] = gpu
                row['type_quota'] = 'allocated'
                result.append(row)
    yt_client.write_table(daily_path, result)
    for elem in result:
        elem['gpu_type'] = elem['type_gpu']
        elem.pop('type_gpu')
        elem['quota_type'] = elem['type_quota']
        elem.pop('type_quota')
        elem['account_id'] += ' ALLOCATED'
    yt_client.write_table(table_path, result)


def get_prices():
    prices = OrderedDict()
    table = yt_client.read_table('//home/capacity_planning/gpu_idle/billing')
    for line in table:
        if line['name'] == 'yp.gpu_tesla_v100.quota':
            prices['gpu_tesla_v100'] = line['unit_price']
            prices['gpu_tesla_v100_nvlink'] = line['unit_price']
        elif line['name'] == 'yt.hahn.gpu.tesla_a100_80g.strong_guarantee.gpu':
            prices['gpu_tesla_a100_80g'] = line['unit_price']
        elif line['name'] == 'yt.hahn.gpu.tesla_a100.strong_guarantee.gpu':
            prices['gpu_tesla_a100'] = line['unit_price']
        elif line['name'] == 'yp.gpu_geforce_1080ti.quota':
            prices['gpu_geforce_1080ti'] = line['unit_price']
        elif line['name'] == 'yp.gpu_tesla_k40.quota':
            prices['gpu_tesla_k40'] = line['unit_price']
        elif line['name'] == 'yp.gpu_tesla_m40.quota':
            prices['gpu_tesla_m40'] = line['unit_price']
    return prices


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--abc_secret', type=str, default=None)
    return parser.parse_args()


def main():
    args = parse_args()
    abc_api = BaseApi(host='https://abc-back.yandex-team.ru', token=args.abc_secret)

    prices = get_prices()
    collect_qouta_free(abc_api, prices)
    collect_allocated(abc_api, prices)
