#!/usr/bin/env python
# coding=utf-8

import json
from collections import OrderedDict
import datetime

import requests
from requests.exceptions import ConnectionError, ChunkedEncodingError
import yt.wrapper as yt
import argparse
import time

from datetime import date

from infra.yp_quota_distributor.lib import hosts_quota
from infra.yp_quota_distributor.lib import constants
from infra.capacity_planning.library.python.yt_tables import create_table
from infra.capacity_planning.library.python.base import retry

default_projects = ['yp-iss-iva-gpu', 'yp-iss-man-gpu', 'yp-iss-sas-gpu', 'yp-iss-vla-gpu']
dev_projects = ['yp-gpu-iva-dev', 'yp-gpu-man-dev', 'yp-gpu-sas-dev', 'yp-gpu-vla-dev']
ml_projects = ['yp-iss-sas-yt-hahn-gpu-yati2', 'yp-iss-vla-yt-hahn-gpu-yati3', 'yp-iss-vla-yt-hahn-gpu-yati4',
               'yp-iss-sas-yt-hahn-gpu-bert', 'yp-iss-vla-yt-hahn-gpu-yati-test']

DC = ['iva', 'myt', 'man', 'sas', 'vla']

base_url = " https://api.wall-e.yandex-team.ru/v1/hosts?project={project}&limit=100&offset={offset}"

cluster = 'hahn'
yt_client = yt.YtClient(cluster)
path_dataset = "<append=true>//home/capacity_planning/gpu_idle/walle_unused_dataset_hour"
path_week = "//home/capacity_planning/gpu_idle/walle_unused_weekly"

DAYS = 60 * 60 * 24

PER_HOUR = 30 * 24


def execute(data):
    """
    Loading hosts parametrs
    :param file: file with all hosts for loading data
    :return: json with all hosts information
    """
    use_format = "json"
    discount_mode = "full"
    text, _ = hosts_quota.calculate_and_format_quota_from_hosts_file(
        data,
        abc_info='TEST',
        abc_segment='TEST',
        console_flag=True,
        discount_mode=discount_mode,
        use_format=use_format,
        ignore_banned_lists=True
    )
    return text


@retry(exceptions=(ConnectionError, ChunkedEncodingError))
def collect():
    """
    Collect all information about scenarios
    :return: collected data
    """
    result = []
    all_projects = default_projects + dev_projects + ml_projects

    datatime = time.mktime(time.strptime(':'.join(str(datetime.datetime.today()).split(":")[:2]), "%Y-%m-%d %H:%M"))
    for project in all_projects:
        offest = 0
        url = base_url.format(project=project, offset=offest)
        response = requests.get(url, verify=False)
        data = response.json()
        row = OrderedDict()
        row["date"] = int(datatime)
        row["project"] = project

        while data["result"]:
            not_work = []
            for host in data["result"]:
                if (not (host["state"] == "assigned" and host["status"] == "ready")) or project == 'yp-iss-vla-yt-hahn-gpu-yati-test':
                    not_work.append(str(host["inv"]))
            offest += 100
            url = base_url.format(project=project, offset=offest)
            response = requests.get(url, verify=False)
            data = response.json()

        print(not_work)
        total_res = json.loads(execute(not_work))
        print(total_res)
        print("\n\n\n")

        total_count = 0
        for gpu, capacity in total_res["quota"]["total"]["gpu_models"].iteritems():
            name = "{}".format(gpu)
            row[name] = float(capacity)
            total_count += float(capacity)
        for host in total_res["hosts"]:
            if not host["quota"]["gpu_model"]:
                try:
                    row["gpu_tesla_a100_80g"] += 8
                except:
                    row["gpu_tesla_a100_80g"] = 8
                total_count += 8

        try:
            row["gpu_tesla_a100_80g"] = float(row["gpu_tesla_a100_80g"])
        except:
            pass
        row["total_gpu_count"] = float(total_count)
        result.append(row)

    return result


def price(data):
    schema = ["date:datetime",
              "cost:double",
              "gpu:double",
              "name:string",
              "project:string",
              ]
    create_table(client=yt_client, path=path_dataset, schema=schema)
    result = []

    table = yt_client.read_table('//home/capacity_planning/gpu_idle/billing')
    prices = OrderedDict()
    for line in table:
        if line["name"] == "yp.gpu_tesla_v100.quota":
            prices["gpu_tesla_v100"] = line["unit_price"] / PER_HOUR
            prices["gpu_tesla_v100_nvlink"] = line["unit_price"] / PER_HOUR
        elif line["name"] == "yt.hahn.gpu.tesla_a100_80g.strong_guarantee.gpu":
            prices["gpu_tesla_a100_80g"] = line["unit_price"] / PER_HOUR
        elif line["name"] == "yt.hahn.gpu.tesla_a100.strong_guarantee.gpu":
            prices["gpu_tesla_a100"] = line["unit_price"] / PER_HOUR
        elif line["name"] == "yp.gpu_geforce_1080ti.quota":
            prices["gpu_geforce_1080ti"] = line["unit_price"] / PER_HOUR

    for line in data:
        for name, value in line.items():
            if "gpu" in name and "total" not in name and "cost" not in name:
                if value is None:
                    continue
                row = OrderedDict()
                row["date"] = line["date"]
                row["cost"] = prices[name] * value
                row["name"] = name
                row["gpu"] = value
                row["project"] = line["project"]
                result.append(row)
    yt_client.write_table(path_dataset, result)


def collect_week():
    schema = [
        "date:datetime",
        "cost:double",
        "gpu:double",
        "name:string",
        "project:string",
    ]
    create_table(client=yt_client, path=path_week, schema=schema)
    today = date.today().strftime("%Y-%m-%d")
    table = yt_client.read_table('//home/capacity_planning/gpu_idle/walle_unused_dataset_hour')
    data = []
    datatime = time.mktime(time.strptime(today, "%Y-%m-%d"))
    for line in table:
        if (datatime - line["date"]) / DAYS <= 10:
            data.append(line)
    yt_client.write_table(path_week, data)


def arg_parse():
    parser = argparse.ArgumentParser()
    parser.add_argument('--secret_token', default=None)
    args = parser.parse_args()
    constants.TOKEN = args.secret_token


def main():
    arg_parse()
    data = collect()
    price(data)
    collect_week()
