#!/usr/bin/env python
# coding=utf-8

import json

from collections import OrderedDict
from datetime import date

import requests
from requests.exceptions import ConnectionError, ChunkedEncodingError
import yt.wrapper as yt
import argparse

from infra.yp_quota_distributor.lib import hosts_quota
from infra.yp_quota_distributor.lib import constants
from infra.capacity_planning.library.python.base import retry

CLUSTER = 'hahn'
TODAY = date.today().strftime("%Y-%m-%d")
YT_PATH = '//home/capacity_planning/reserves/hosts/' + TODAY
yt_client = yt.YtClient(CLUSTER)


def execute(data):
    """
    Loading hosts parametrs
    :param file: file with all hosts for loading data
    :return: json with all hosts information
    """
    use_format = "json"
    discount_mode = "full"
    text, _ = hosts_quota.calculate_and_format_quota_from_hosts_file(
        data,
        abc_info='TEST',
        abc_segment='TEST',
        console_flag=True,
        discount_mode=discount_mode,
        use_format=use_format,
        ignore_banned_lists=True
    )
    return text


def create_table():
    """
    Creating table in YT
    :return: None
    """
    yt_client.create("table", YT_PATH,
                     ignore_existing=True,
                     attributes={
                         "schema":
                             [
                                 {"name": "id", "type": "int64"},
                                 {"name": "status", "type": "string"},
                                 {"name": "ticket", "type": "string"},
                                 {"name": "issuer", "type": "string"},
                                 {"name": "responsible", "type": "string"},
                                 {"name": "ticket_created_by", "type": "string"},
                                 {"name": "hosts", "type": "int64"},
                                 {"name": "target_project_id", "type": "string"},
                                 {"name": "invs", "type": "any"},
                                 {"name": "dc", "type": "string"},
                                 {"name": "cpu_total", "type": "double"},
                                 {"name": "cpu_remaining", "type": "double"},
                                 {"name": "memory_total", "type": "double"},
                                 {"name": "memory_remaining", "type": "double"},
                                 {"name": "hdd_total", "type": "double"},
                                 {"name": "hdd_remaining", "type": "double"},
                                 {"name": "ssd_total", "type": "double"},
                                 {"name": "ssd_remaining", "type": "double"},
                                 {"name": "io_hdd_total", "type": "double"},
                                 {"name": "io_hdd_remaining", "type": "double"},
                                 {"name": "io_ssd_total", "type": "double"},
                                 {"name": "io_ssd_remaining", "type": "double"},
                                 {"name": "gpu_tesla_v100_total", "type": "double"},
                                 {"name": "gpu_tesla_v100_remaining", "type": "double"},
                                 {"name": "gpu_tesla_v100_nvlink_total", "type": "double"},
                                 {"name": "gpu_tesla_v100_nvlink_remaining", "type": "double"},
                                 {"name": "gpu_tesla_a100_total", "type": "double"},
                                 {"name": "gpu_tesla_a100_remaining", "type": "double"},
                                 {"name": "gpu_tesla_a100_80g_total", "type": "double"},
                                 {"name": "gpu_tesla_a100_80g_remaining", "type": "double"},
                                 {"name": "gpu_geforce_1080ti_total", "type": "double"},
                                 {"name": "gpu_geforce_1080ti_remaining", "type": "double"},
                             ]
                     }
                     )


dev_projects = ['yp-iss-iva-dev', 'yp-iss-myt-dev', 'yp-iss-man-dev', 'yp-iss-sas-dev', 'yp-iss-vla-dev']
default_projects = ['yp-iss-iva', 'yp-iss-myt', 'yp-iss-man', 'yp-iss-sas', 'yp-iss-vla']

gpu_default_projects_yp = ['yp-iss-iva-gpu', 'yp-iss-myt-gpu', 'yp-iss-man-gpu', 'yp-iss-sas-gpu', 'yp-iss-vla-gpu']
gpu_dev_projects_yp = ['yp-gpu-iva-dev', 'yp-gpu-myt-dev', 'yp-gpu-man-dev', 'yp-gpu-sas-dev', 'yp-gpu-vla-dev']

gencfg_proecjts = ['rtc-mtn']

DC = ['iva', 'myt', 'man', 'sas', 'vla']

base_url = "https://api.wall-e.yandex-team.ru/v1/scenarios?&limit=100&offset={offset}"
tag_url = "https://api.wall-e.yandex-team.ru/v1/projects/{proj}"


@retry(exceptions=(ConnectionError, ChunkedEncodingError))
def collect():
    """
    Collect all information about scenarios
    :return: collected data
    """
    offest = 0
    result = []
    url = base_url.format(offset=offest)
    response = requests.get(url, verify=False)
    data = response.json()

    while data["result"]:
        for scenar in data["result"]:
            if scenar["status"] not in ["canceled", "canceling", "rejected", "finished", "finishing"]:
                row = OrderedDict()
                try:
                    row["target_project_id"] = scenar["script_args"]["target_project_id"]

                    url_tag = tag_url.format(proj=row["target_project_id"])
                    response_tag = requests.get(url_tag, verify=False)
                    tags = response_tag.json()
                    tags = tags['tags']
                    if not ('rtc' in tags and 'yt' not in tags):
                        continue
                except:
                    continue

                row["id"] = scenar["scenario_id"]
                row["status"] = scenar["status"]
                row["ticket"] = scenar["ticket_key"]
                row["issuer"] = scenar["issuer"]
                try:
                    row["responsible"] = scenar["labels"]["responsible"]
                except:
                    pass
                try:
                    row["ticket_created_by"] = scenar["labels"]["ticket_created_by"]
                except:
                    pass
                row["hosts"] = len(scenar["hosts"])

                total = []
                remaining = []

                for host in scenar["hosts"]:
                    total.append(str(host["inv"]))
                    if host["status"] != 'done':
                        remaining.append(str(host["inv"]))
                row["invs"] = [int(elem) for elem in total]

                capacity_total = json.loads(execute(total))

                capacity_remaining = json.loads(execute(remaining))

                row["cpu_total"] = float(capacity_total["quota"]["total"]["cpu"])
                row["memory_total"] = float(capacity_total["quota"]["total"]["memory"])
                row["ssd_total"] = float(capacity_total["quota"]["total"]["ssd"])
                row["hdd_total"] = float(capacity_total["quota"]["total"]["hdd"])
                row["io_hdd_total"] = float(capacity_total["quota"]["total"]["io_hdd"])
                row["io_ssd_total"] = float(capacity_total["quota"]["total"]["io_ssd"])
                for gpu, capacity in capacity_total["quota"]["total"]["gpu_models"].iteritems():
                    name = "{}_total".format(gpu)
                    row[name] = float(capacity)

                row["cpu_remaining"] = float(capacity_remaining["quota"]["total"]["cpu"])
                row["memory_remaining"] = float(capacity_remaining["quota"]["total"]["memory"])
                row["ssd_remaining"] = float(capacity_remaining["quota"]["total"]["ssd"])
                row["hdd_remaining"] = float(capacity_remaining["quota"]["total"]["hdd"])
                row["io_hdd_remaining"] = float(capacity_remaining["quota"]["total"]["io_hdd"])
                row["io_ssd_remaining"] = float(capacity_remaining["quota"]["total"]["io_ssd"])
                for gpu, capacity in capacity_remaining["quota"]["total"]["gpu_models"].iteritems():
                    name = "{}_remaining".format(gpu)
                    row[name] = float(capacity)

                for elem in DC:
                    if elem in scenar["script_args"]["target_project_id"]:
                        row["dc"] = elem
                        break

                result.append(row)
        offest += 100
        url = base_url.format(offset=offest)
        response = requests.get(url, verify=False)
        data = response.json()

    return result


def arg_parse():
    parser = argparse.ArgumentParser()
    parser.add_argument('--secret_token', default=None)
    args = parser.parse_args()
    constants.TOKEN = args.secret_token


def main():
    arg_parse()
    create_table()
    data = collect()
    yt_client.write_table(YT_PATH, data)


if __name__ == '__main__':
    main()
