#!/bin/sh
################################################################################
#                                                                              #
# This script is a part of yandex-cauth package. It does the following:        #
# 1. Imports all groups and users from /etc/security/*.conf, allowed to login  #
# 2. Downloads admin's ssh keys to /root/.ssh/authorized_keys                  #
# 3. Downloads user's ssh public keys from Cauth API                           #
# 4. Creates .ssh user's dirs if does not exists                               #
# 5. Compares new user's ssh keys from CAuth with local ones and               #
#    install new ones if differs                                               #
#                                                                              #
#                                               zhmurov@yandex-team.ru         #
#                                                                              #
################################################################################

# Import file with CAuth functions
if [ -s /etc/cauth/cauth_functions ]; then
    . /etc/cauth/cauth_functions
else
    exit 1
fi

# Do we have sufficient privileges?
is_root || p_err "You need to be root to run $0"


# Create working directories if does not exist
create_working_dirs || p_err "Cannot create working directories"


# Get groups and logins from access lists allowed to login
allowed_groups_list=$(awk -F":" '/^+.*dpt_.*:/ {print $2}' ${static_access_list} 2>/dev/null)
allowed_logins_list=$(awk -F":" '/^+ / {print $2}' ${static_access_list} 2>/dev/null | grep -Ev "( root |dpt_)" 2>/dev/null)


#
# import_keys_from_fetched_file()
#  This function searches public_ssh_keys for login "$1" in file "$2" and installs it
#  to login's homedir/.ssh/authorized_keys
#
import_keys_from_fetched_file()
{
    local _login _group _old_md5 _new_md5 _home_dir _home_dir_has_wrong_perms _ssh_dir
    local _ssh_dir_has_wrong_perms _ssh_key_file _ssh_key2_file _ssh_key_file_has_wrong_perms
    local _tmp_key_file _tmp_new_key_file _tmp_ssh_dir _keys_file _new_key _cur_key

    if [ "$#" -lt 2 ]; then
        return 1
    fi

    _login="$1"
    _keys_file="$2"
    _group="$(id -g -n ${_login} 2>/dev/null)"
    _home_dir="$(getent passwd ${_login} | awk -F: '{print $6}')"
    _ssh_dir="${_home_dir}/.ssh"
    _ssh_key_file="${_ssh_dir}/authorized_keys"
    _ssh_key2_file="${_ssh_dir}/authorized_keys2"
    _home_dir_has_wrong_perms="0"
    _ssh_dir_has_wrong_perms="0"
    _ssh_key_file_has_wrong_perms="0"

    _home_dir_mode=0755
    if [ ! -z "${home_dir_mode+x}" ] ; then
        _home_dir_mode=${home_dir_mode}
    fi

    # fail if cannot resolve ${_login}
    if ! id "${_login}" >/dev/null 2>&1 ; then
        return 1
    fi

    # fail if ${_group} is empty
    if test -z "${_group}" ; then
        return 1
    fi

    # fail if cannot resolve ${_home_dir}
    if test -z "${_home_dir}" ; then
        return 1
    fi


    # Unprivileged users MUST NOT have authorized_keys2, remove it if found
    if [ -f "${_ssh_key2_file}" ] ; then
        chattr -i ${_ssh_key2_file} 2>/dev/null
        rm -f ${_ssh_key2_file} 2>/dev/null
    fi


    # we need this to compare current ssh key and new one
    _cur_key="$(cat ${_ssh_key_file} 2>/dev/null)"
    _new_key="$(awk -F" : " "/^${_login} :/ {print \$2}" ${_keys_file})"


    # Detect incorrect permissions/owner for ${_home_dir}
    if [ -d "${_home_dir}" -a ! -L "${_home_dir}" ] ; then
        if [ "$(stat ${stat_ugp_opts} ${_home_dir})" != "${_login}${_group}${_home_dir_mode##0}" ] ; then
            _home_dir_has_wrong_perms=1
        fi
    fi


    # Detect incorrect permissions/owner for ${_ssh_dir}
    if [ -d "${_ssh_dir}" -a ! -L "${_ssh_dir}" ] ; then
        if [ "$(stat ${stat_ugp_opts} ${_ssh_dir})" != "${_login}${_group}700" ] ; then
            _ssh_dir_has_wrong_perms=1
        fi
    fi


    # Detect incorrect permissions/owner for ${_ssh_key_file}
    if [ -f "${_ssh_key_file}" -a ! -L "${_ssh_key_file}" ] ; then
        if [ "$(stat ${stat_ugp_opts} ${_ssh_key_file})" != "${_login}${_group}600" ] ; then
            _ssh_key_file_has_wrong_perms=1
        fi
    fi


    # If ${_home_dir} does not exist, create it and copy configs from /etc/skel
    # If incorrect privileges detected - fix it
    if [ ! -d "${_home_dir}" ] ; then
        install -d -m${_home_dir_mode} -o ${_login} -g ${_group} ${_home_dir} || \
            p_err "Cannot create ${_home_dir}"

        install -m0644 -o ${_login} -g ${_group} /etc/skel/.[a-z]* ${_home_dir}/ 2>/dev/null

    elif [ "${_home_dir_has_wrong_perms}" -eq "1" ] ; then
        install -d -m${_home_dir_mode} -o ${_login} -g ${_group} ${_home_dir} || \
            p_err "Cannot change permissions to ${_home_dir}"

    fi


    # If ${_ssh_dir} does not exist, create it with with appropriate permissions
    # If incorrect privileges detected - fix it
    if [ ! -d "${_ssh_dir}" -o -L "${_ssh_dir}" -o "${_ssh_dir_has_wrong_perms}" -eq "1" ] ; then
        # drop privileges to ${_login}
        if ! su -s /bin/sh -c "install -d -m0700 -o ${_login} -g ${_group} ${_ssh_dir}" ${_login} ; then
            p_err "Cannot create (or fix permission of) ${_ssh_dir}" "not_fatal"
            return
        fi
    fi


    # Install brand new authorized_keys if current ssh key and one from CAuth differs,
    # or incorrect permissions/owner detected
    if [ ! -z "${_new_key}" -a "${_cur_key}" != "${_new_key}" -o "${_ssh_key_file_has_wrong_perms}" -eq "1" ] ; then

        _tmp_new_key_file=$(mktemp ${private_tmp_dir}/tmp_new_key_file.XXXXXX)
        if ! echo "${_new_key}" > ${_tmp_new_key_file} ; then
            clean_garbage ${_tmp_new_key_file}
            p_err "Cannot save new key file to ${_tmp_new_key_file}" "not_fatal"
            return
        fi

        # change ownership only available for privileged user
        if ! chown ${_login}:${_group} ${_tmp_new_key_file} ; then
            clean_garbage ${_tmp_new_key_file}
            p_err "Cannot change owner of ${_tmp_new_key_file} to ${_login}:${_group}" "not_fatal"
            return
        fi

        # drop privileges to ${_login}
        if ! su -s /bin/sh -c "install -m0600 ${_tmp_new_key_file} ${_ssh_key_file}" ${_login} ; then
            clean_garbage ${_tmp_new_key_file}
            p_err "Cannot install ${_tmp_new_key_file} to ${_ssh_key_file} with ${_login} privileges" "not_fatal"
            return
        fi

        rm -f ${_tmp_new_key_file}
    fi
}


#
# install_root_keys()
#  This function downloads root's authorized_keys from ${adminkeys_url} and
#  installs it to root's homedir/.ssh/authorized_keys
#
install_root_keys()
{
    # Correct permissions for root's .ssh dir
    if [ "$(stat ${stat_ugp_opts} /root/.ssh)" != "rootroot700" ] ; then
        install -d -m0700 -o root -g root /root/.ssh
    fi

    fetch_file ${root_keys_file} ${adminkeys_url}
    chmod 0600 ${root_keys_file}
}


# install_user_keys()
#  This function downloads ssh_public_keys of all users for current server
#  and installs (via import_keys_from_fetched_file() function) ssh_piblic_key
#  for each user
#
install_user_keys()
{
    local _user _user_list

    fetch_file ${userkeys_file} ${userkeys_url}

    _user_list=$(awk -F: '!/^#/ {print $1}' ${userkeys_file} | sort | uniq)
    for _user in ${_user_list} ; do
        import_keys_from_fetched_file ${_user} ${userkeys_file}
    done
}


# install_static_user_keys()
#  There are possibility to allow access to server not just via
#  CAUTH api/interface (/etc/security/yandex-access-custom.conf), but 
#  via static/hardcoded users in special file (/etc/security/yandex-access.conf),
#  so function install_static_user_keys() gets such users from /etc/security/yandex-access.conf,
#  downloads keys for each user from that file and does exactly same thing as install_user_keys does
#
install_static_user_keys()
{
    local _user _user_list _y

    if [ ! -z "$allowed_groups_list" -o ! -z "$allowed_logins_list" ] ; then

        for _y in ${allowed_groups_list} ${allowed_logins_list} ; do
            fetch_file ${cauth_spool}/${_y} ${userkeys_url}/${_y}

            _user_list=$(awk -F: '!/^#/ {print $1}' ${cauth_spool}/${_y} | sort | uniq)
            for _user in ${_user_list} ; do
                import_keys_from_fetched_file ${_user} ${cauth_spool}/${_y}
            done
        done

    fi
}


install_user_keys
install_static_user_keys
install_root_keys
