package app

import (
	"context"
	"fmt"
	"net"
	"net/http"
	"os"
	"syscall"

	"a.yandex-team.ru/infra/cauth/agent/linux/cauth-agent/internal/config"
	"a.yandex-team.ru/infra/cauth/agent/linux/cauth-agent/internal/nssdb"
	"a.yandex-team.ru/library/go/core/log"
)

type App struct {
	cfg  *config.Config
	http *http.Server
	log  log.Logger
	db   *nssdb.Storage
}

func NewApp(cfg *config.Config, l log.Logger) (app *App, err error) {
	app = &App{
		cfg: cfg,
		log: l,
		db:  nssdb.NewStorage(cfg.NSS, l),
	}

	app.http = &http.Server{
		Handler: newRouter(app),
	}
	return app, nil
}

func (a *App) Start() error {
	if err := a.db.Reload(); err != nil {
		return fmt.Errorf("can't initialize db: %w", err)
	}

	a.log.Info("start http API", log.String("addr", a.cfg.Addr))

	_ = syscall.Unlink(a.cfg.Addr)
	listener, err := net.Listen("unix", a.cfg.Addr)
	if err != nil {
		return err
	}

	defer func() {
		_ = listener.Close()
	}()

	_ = os.Chmod(a.cfg.Addr, 0666)
	return a.http.Serve(listener)
}

func (a *App) Reload() error {
	return a.db.Reload()
}

func (a *App) Shutdown(ctx context.Context) error {
	return a.http.Shutdown(ctx)
}
