package app

import (
	"fmt"
	"net/http"
	"strconv"

	"github.com/go-chi/chi/v5"

	"a.yandex-team.ru/library/go/httputil/middleware/recovery"
)

var (
	okRsp = []byte(`OK`)
)

func newRouter(a *App) *chi.Mux {
	r := chi.NewRouter()
	r.Use(recovery.New(recovery.WithLogger(a.log.Structured())))
	r.Use(logRequest(a.log))

	r.Route("/nss/v1", func(r chi.Router) {
		r.Get("/user/{uid:[0-9]+}", a.userByUIDHandler)
		r.Get("/user/{login}", a.userByLoginHandler)
		r.Get("/group/{gid:[0-9]+}", a.groupByGIDHandler)
		r.Get("/group/{name}", a.groupByNameHandler)
		r.Get("/groups/{login}", a.userGroupsHandler)
	})

	r.Get("/ping", a.pingHandler)
	return r
}

func (a *App) pingHandler(w http.ResponseWriter, _ *http.Request) {
	_, _ = w.Write(okRsp)
}

func (a *App) userByUIDHandler(w http.ResponseWriter, r *http.Request) {
	rawUID := chi.URLParam(r, "uid")
	if rawUID == "" {
		http.Error(w, "no uid provided", http.StatusBadRequest)
		return
	}

	uid, err := strconv.ParseUint(rawUID, 10, 32)
	if err != nil {
		http.Error(w, fmt.Sprintf("invalid uid (%q): %v", rawUID, err), http.StatusBadRequest)
		return
	}

	user, ok := a.db.UserByUID(uint32(uid))
	if !ok {
		http.Error(w, "user not found", http.StatusNotFound)
		return
	}

	w.Header().Set("Content-Type", "application/x-protobuf")
	_, _ = w.Write(user)
}

func (a *App) userByLoginHandler(w http.ResponseWriter, r *http.Request) {
	login := chi.URLParam(r, "login")
	if login == "" {
		http.Error(w, "no login provided", http.StatusBadRequest)
		return
	}

	user, ok := a.db.UserByLogin(login)
	if !ok {
		http.Error(w, "user not found", http.StatusNotFound)
		return
	}

	w.Header().Set("Content-Type", "application/x-protobuf")
	_, _ = w.Write(user)
}

func (a *App) userGroupsHandler(w http.ResponseWriter, r *http.Request) {
	login := chi.URLParam(r, "login")
	if login == "" {
		http.Error(w, "no login provided", http.StatusBadRequest)
		return
	}

	groups, ok := a.db.UserGroups(login)
	if !ok {
		http.Error(w, "user not found", http.StatusNotFound)
		return
	}

	w.Header().Set("Content-Type", "application/x-protobuf")
	_, _ = w.Write(groups)
}

func (a *App) groupByGIDHandler(w http.ResponseWriter, r *http.Request) {
	rawGID := chi.URLParam(r, "gid")
	if rawGID == "" {
		http.Error(w, "no gid provided", http.StatusBadRequest)
		return
	}

	gid, err := strconv.ParseUint(rawGID, 10, 32)
	if err != nil {
		http.Error(w, fmt.Sprintf("invalid gid (%q): %v", rawGID, err), http.StatusBadRequest)
		return
	}

	group, ok := a.db.GroupByGID(uint32(gid))
	if !ok {
		http.Error(w, "group not found", http.StatusNotFound)
		return
	}

	w.Header().Set("Content-Type", "application/x-protobuf")
	_, _ = w.Write(group)
}

func (a *App) groupByNameHandler(w http.ResponseWriter, r *http.Request) {
	name := chi.URLParam(r, "name")
	if name == "" {
		http.Error(w, "no name provided", http.StatusBadRequest)
		return
	}

	group, ok := a.db.GroupByName(name)
	if !ok {
		http.Error(w, "group not found", http.StatusNotFound)
		return
	}

	w.Header().Set("Content-Type", "application/x-protobuf")
	_, _ = w.Write(group)
}
