package commands

import (
	"context"
	"fmt"
	"os"
	"os/signal"
	"syscall"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/infra/cauth/agent/linux/cauth-agent/internal/app"
	"a.yandex-team.ru/infra/cauth/agent/linux/cauth-agent/internal/config"
	"a.yandex-team.ru/library/go/core/log"
)

var startCmd = &cobra.Command{
	Use:          "start",
	SilenceUsage: true,
	Short:        "Start CAuth-agent",
	RunE: func(cmd *cobra.Command, args []string) error {
		instance, err := app.NewApp(cfg, logger)
		if err != nil {
			return fmt.Errorf("failed to create application: %w", err)
		}

		errChan := make(chan error, 1)
		okChan := make(chan struct{}, 1)
		go func() {
			if err := instance.Start(); err != nil {
				errChan <- fmt.Errorf("failed to start application: %w", err)
			} else {
				okChan <- struct{}{}
			}
		}()

		sigChan := make(chan os.Signal, 1)
		signal.Notify(sigChan, syscall.SIGINT, syscall.SIGTERM, syscall.SIGUSR1)

		defer logger.Info("CAuth-agent stopped")

		for {
			select {
			case s := <-sigChan:
				switch s {
				case syscall.SIGUSR1:
					logger.Info("reloading")
					if err := instance.Reload(); err != nil {
						logger.Error("reload failed", log.Error(err))
					} else {
						logger.Info("reloaded")
					}
				default:
					logger.Info("shutting down gracefully by signal")

					return func() error {
						ctx, cancel := context.WithTimeout(context.Background(), config.ShutdownDeadline)
						defer cancel()

						return instance.Shutdown(ctx)
					}()
				}
			case <-okChan:
				return nil
			case err := <-errChan:
				return err
			}
		}
	},
}

func init() {
	rootCmd.AddCommand(startCmd)
}
