package config

import (
	"context"
	"fmt"
	"strings"
	"time"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/file"
	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"

	"a.yandex-team.ru/library/go/core/log"
)

const (
	ShutdownDeadline = 1 * time.Minute
)

type LogConfig struct {
	Level  string `yaml:"level"`
	Output string `yaml:"output"`
}

type NSSConfig struct {
	DBPath string `yaml:"dbPath"`
}

type Config struct {
	Log  LogConfig `yaml:"log"`
	Addr string    `yaml:"addr"`
	NSS  NSSConfig `yaml:"nss"`
}

func NewConfig(configPath string) (*Config, error) {
	cfg := &Config{
		Log: LogConfig{
			Level:  log.DebugString,
			Output: "stdout",
		},
		NSS: NSSConfig{
			DBPath: "/etc/cauth/cache.fb",
		},
		Addr: "/run/cauth-agent.sock",
	}

	if configPath == "" {
		return cfg, nil
	}

	loader := confita.NewLoader(file.NewBackend(configPath))
	if err := loader.Load(context.Background(), cfg); err != nil {
		return nil, fmt.Errorf("failed to parse config: %w", err)
	}

	return cfg, nil
}

func (l *LogConfig) ZapConfig() zap.Config {
	var lvl zapcore.Level
	switch strings.ToLower(l.Level) {
	case log.TraceString:
		lvl = zapcore.DebugLevel
	case log.DebugString:
		lvl = zapcore.DebugLevel
	case log.InfoString:
		lvl = zapcore.InfoLevel
	case log.WarnString:
		lvl = zapcore.WarnLevel
	case log.ErrorString:
		lvl = zapcore.ErrorLevel
	case log.FatalString:
		lvl = zapcore.FatalLevel
	default:
		lvl = zapcore.InfoLevel
	}

	return zap.Config{
		Level:            zap.NewAtomicLevelAt(lvl),
		Encoding:         "kv",
		OutputPaths:      []string{l.Output},
		ErrorOutputPaths: []string{"stderr"},
		EncoderConfig: zapcore.EncoderConfig{
			MessageKey:     "msg",
			LevelKey:       "level",
			TimeKey:        "ts",
			NameKey:        "logger",
			LineEnding:     zapcore.DefaultLineEnding,
			EncodeLevel:    zapcore.CapitalLevelEncoder,
			EncodeTime:     zapcore.ISO8601TimeEncoder,
			EncodeDuration: zapcore.SecondsDurationEncoder,
			EncodeCaller:   zapcore.ShortCallerEncoder,
		},
	}
}
