import argparse
import os
import time
import urllib.parse

import requests_unixsocket
import yaml
from infra.rtc.juggler.bundle.util import WalleCompatEvent
from juggler.bundles import as_check, Status

DEFAULT_USERD_CONF = '/etc/cauth/yandex-cauth-userd.yaml'
DEFAULT_CONFIG = {
    'daemonConfig': {
        'pidFile': '/run/yandex-cauth-userd.pid',
        'socketPath': '/run/yandex-cauth-userd.sock',
    },
    'repoConfig': {
        'path': '/var/cache/yandex-cauth-userd',
    },
}
CURRENT_AGE_THRESHOLD = 12 * 3600
CHECK_NAME = 'cauth_userd'


def read_config(path=DEFAULT_USERD_CONF, open_func=open):
    try:
        with open_func(path) as f:
            return yaml.safe_load(f), None
    except Exception as e:
        return None, f'failed to read config {path}: {str(e)}'


def get_pidfile_from_config(config):
    return config.get('daemonConfig', DEFAULT_CONFIG['daemonConfig']).get('pidFile',
                                                                          DEFAULT_CONFIG['daemonConfig']['pidFile'])


def get_socket_path_from_config(config):
    return config.get('daemonConfig', DEFAULT_CONFIG['daemonConfig']).get('socketPath',
                                                                          DEFAULT_CONFIG['daemonConfig']['socketPath'])


def get_repo_path_from_config(config):
    return config.get('repoConfig', DEFAULT_CONFIG['repoConfig']).get('path', DEFAULT_CONFIG['repoConfig']['path'])


def get_pid(path, open_func=open):
    try:
        with open_func(path) as f:
            return f.read(32).strip(), None
    except Exception as e:
        return None, f'failed to read pid file {path}: {str(e)}'


def pid_running(pid, isdir=os.path.isdir):
    try:
        return isdir(f'/proc/{pid}'), None
    except Exception as e:
        return None, f'failed to check is pid {pid} running: {str(e)}'


def group_exists(sock_path, name, session=requests_unixsocket.Session):
    with session() as s:
        try:
            req_path = f'http+unix://{urllib.parse.quote(sock_path, safe="")}/nss/v1/group/{name}'
            rsp = s.get(req_path)
            if rsp.status_code == 200:
                return True, None
            elif rsp.status_code == 404:
                return False, None
            else:
                return False, f'unexpected status code {rsp.status_code} from {req_path}'
        except Exception as e:
            return False, f'failed to resolve group {name} via {sock_path}: {str(e)}'


def get_target_timestamp(link, isdir=os.path.isdir, readlink=os.readlink, stat=os.stat):
    try:
        target = readlink(link)
    except Exception as e:
        return None, f'failed to read link {link}: {str(e)}'
    try:
        target_dir = isdir(target)
    except Exception as e:
        return None, f'failed to check {target} is directory: {str(e)}'
    if target_dir is not True:
        return None, f'{target} is not a directory'
    try:
        s = stat(target)
        return s.st_mtime, None
    except Exception as e:
        return None, f'failed to stat {target}: {str(e)}'


def parse_args(args):
    parser = argparse.ArgumentParser()
    parser.add_argument('--userd-config-path', default=None, dest='userd_config_path',
                        help=f'Path to yandex-cauth-userd config (default: "{DEFAULT_USERD_CONF}"')
    return parser.parse_args(args)


def check_userd_running(config, open_func=open, isdir=os.path.isdir):
    pidfile = get_pidfile_from_config(config)
    pid, err = get_pid(pidfile, open_func)
    if err is not None:
        return Status.CRIT, f'Failed to check yandex-cauth-userd running: {err}'
    ok, err = pid_running(pid, isdir)
    if err is not None:
        return Status.CRIT, f'Failed to check yandex-cauth-userd running: {err}'
    if ok is True:
        return Status.OK, None
    else:
        return Status.CRIT, 'yandex-cauth-userd is not running'


def check_userd_resolver(config, session=requests_unixsocket.Session):
    sock_path = get_socket_path_from_config(config)
    ok, err = group_exists(sock_path, 'dpt_yandex', session)
    if err is not None:
        return Status.CRIT, f'Failed to resolve dpt_yandex group: {err}'
    if ok is True:
        return Status.OK, None
    else:
        return Status.CRIT, 'Group dpt_yandex does not exist'


def check_userd_current(config, readlink=os.readlink, stat=os.stat, isdir=os.path.isdir):
    repo_path = get_repo_path_from_config(config)
    current_path = os.path.join(repo_path, 'current')
    ts, err = get_target_timestamp(current_path, isdir=isdir, readlink=readlink, stat=stat)
    if err is not None:
        return Status.CRIT, f'Failed to check current timestamp: {err}'
    now = time.time()
    if now - ts > CURRENT_AGE_THRESHOLD:
        return Status.CRIT, f'yandex-cauth-userd data at {current_path} last updated more than {CURRENT_AGE_THRESHOLD} seconds ago'
    else:
        return Status.OK, None


def run_next(status, err):
    return status == Status.OK and err is None


def run_check(args, exists=os.path.exists):
    try:
        check_args = parse_args(args)
        # use defaults if config not provided explicitly and does not exists
        if check_args.userd_config_path is None and not exists(DEFAULT_USERD_CONF):
            config = DEFAULT_CONFIG
            err = None
        else:
            config, err = read_config(check_args.userd_config_path)
        if err is not None:
            return WalleCompatEvent(Status.CRIT, err)

        status, err = check_userd_running(config)
        if not run_next(status, err):
            return WalleCompatEvent(status, err)

        status, err = check_userd_resolver(config)
        if not run_next(status, err):
            return WalleCompatEvent(status, err)

        status, err = check_userd_current(config)
        if not run_next(status, err):
            return WalleCompatEvent(status, err)

    except Exception as e:
        return WalleCompatEvent(Status.CRIT, str(e))
    return WalleCompatEvent(Status.OK, 'ok')


@as_check(name=CHECK_NAME)
def juggler_check(*args):
    return run_check(args)
