#include "cache.h"

#include <util/system/fstat.h>

#include <utility>

namespace NCAuth::NSS {

    namespace {
        const char* marker = "DEADBEEF";
        constexpr size_t markerLen = std::string_view("DEADBEEF").length();
    }

    TCache::TCache(TString cachePath, bool stale) :
            stale(stale),
            inode(0),
            cachePath(std::move(cachePath)),
            cache(nullptr) {
    };

    const NCAuth::NFlatCache::User* TCache::UserByUid(ui32 uid) {
        Remap();
        TReadGuard guard(mu);

        return cache->users()->LookupByKey(uid);
    }

    const NCAuth::NFlatCache::User* TCache::UserByLogin(const char* login) {
        Remap();
        TReadGuard guard(mu);

        auto u = cache->user_names()->LookupByKey(login);
        if (!u) {
            return nullptr;
        }

        return u->user();
    }

    const NCAuth::NFlatCache::Group* TCache::GroupByGid(ui32 gid) {
        Remap();
        TReadGuard guard(mu);

        return cache->groups()->LookupByKey(gid);
    }

    const NCAuth::NFlatCache::Group* TCache::GroupByName(const char* groupName) {
        Remap();
        TReadGuard guard(mu);

        auto g = cache->group_names()->LookupByKey(groupName);
        if (!g) {
            return nullptr;
        }

        return g->group();
    }

    void TCache::Remap(bool force) {
        ui64 curInode = TFileStat(cachePath).INode;

        {
            TReadGuard guard(mu);

            if (!force && mCache.Get() && mCache->Ptr() && (curInode == inode || stale)) {
                return;
            }
        }

        {
            TWriteGuard guard(mu);

            mCache.Reset(new TFileMap{cachePath, TMemoryMapCommon::oRdOnly | TMemoryMapCommon::oPopulate});
            size_t size = mCache->GetFile().GetLength();
            if (size < markerLen * 2) {
                ythrow yexception() << "corrupted DB: invalid size (" << size << " < " << (markerLen * 2) << ")";
            }

            mCache->Map(0, size);
            const char* data = reinterpret_cast<const char*>(mCache->Ptr());
            if (memcmp(data, marker, markerLen) != 0) {
                ythrow yexception() << "corrupted DB: start marker not found";
            }

            if (memcmp(data + size - markerLen, marker, markerLen) != 0) {
                ythrow yexception() << "corrupted DB: end marker not found";
            }

            cache = NCAuth::NFlatCache::GetCache(data+markerLen);
            inode = curInode;
        }
    }

    TUserIterator::TUserIterator(const TString& cachePath) :
            TCache(cachePath, true),
            started(false),
            stopped(false) {
    }

    bool TUserIterator::Next() {
        Remap();
        TGuard guard{mu};

        if (stopped) {
            return false;
        }

        if (!started) {
            started = true;
            begin = cache->users()->cbegin();
            end = cache->users()->cend();
        } else {
            ++begin;
        }

        return begin != end;
    }

    void TUserIterator::Stop() {
        TGuard guard{mu};

        stopped = true;
    }

    const NCAuth::NFlatCache::User* TUserIterator::User() {
        TGuard guard{mu};

        return *begin;
    }

    TGroupIterator::TGroupIterator(const TString& cachePath) :
            TCache(cachePath, true),
            started(false),
            stopped(false) {
    }

    bool TGroupIterator::Next() {
        Remap();
        TGuard guard{mu};

        if (stopped) {
            return false;
        }

        if (!started) {
            started = true;
            begin = cache->groups()->cbegin();
            end = cache->groups()->cend();
        } else {
            ++begin;
        }

        return begin != end;
    }

    const NCAuth::NFlatCache::Group* TGroupIterator::Group() {
        TGuard guard{mu};

        return *begin;
    }

    void TGroupIterator::Stop() {
        TGuard guard{mu};

        stopped = true;
    }

} // namespace NCAuth::NSS
