#include <grp.h>
#include <nss.h>
#include <pwd.h>
#include <syslog.h>
#include <cstdlib>

#include <util/system/filemap.h>

#include "cache.h"
#include <infra/cauth/agent/linux/flatcache/cache.fbs.h>
#include <infra/cauth/agent/linux/nss_utils/utils.h>


using namespace NCAuth::NSS;

#define SYSLOG(fmt, ...)                                     \
  do {                                                       \
      openlog("nss_cauth", LOG_PID|LOG_PERROR, LOG_USER);    \
      syslog(LOG_ERR, fmt, ##__VA_ARGS__);                   \
      closelog();                                            \
  } while (0)


namespace {
    const __uid_t kMinUID = 20000;
    const __gid_t kMinGID = 20000;
    const __gid_t kMaxGID = (gid_t) -2;
    const ui32 kMaxGroups = 1000;
    const size_t kMinGroupBufSize = 4096;
    const char* kCachePath = "/etc/cauth/cache.fb";

    THolder<TUserIterator> passwdIter = nullptr;
    THolder<TGroupIterator> groupIter = nullptr;

    nss_status UserForNSS(const NCAuth::NFlatCache::User* user, struct passwd* result,
                          char* buffer, size_t buflen, int* errnop) {
        result->pw_uid = user->uid();
        result->pw_gid = user->gid();
        result->pw_passwd = (char*) "*";

        NUtils::TBufferManager buf(buffer, buflen);
        if (!buf.AppendString(user->login()->string_view(), &result->pw_name, errnop)) {
            return NSS_STATUS_TRYAGAIN;
        }

        if (!buf.AppendString(user->login()->string_view(), &result->pw_gecos, errnop)) {
            return NSS_STATUS_TRYAGAIN;
        }

        if (!buf.AppendString(user->home_dir()->string_view(), &result->pw_dir, errnop)) {
            return NSS_STATUS_TRYAGAIN;
        }

        if (!buf.AppendString(user->shell()->string_view(), &result->pw_shell, errnop)) {
            return NSS_STATUS_TRYAGAIN;
        }

        return NSS_STATUS_SUCCESS;
    }

    nss_status GroupForNSS(const NCAuth::NFlatCache::Group* group, struct group* result,
                           char* buffer, size_t buflen, int* errnop) {
        result->gr_gid = group->gid();
        result->gr_passwd = (char*) "*";

        NUtils::TBufferManager buf(buffer, buflen);

        if (!buf.AppendString(group->name()->string_view(), &result->gr_name, errnop)) {
            return NSS_STATUS_TRYAGAIN;
        }

        char** bufp;
        bufp = (char**) buf.Reserve(sizeof(char*) * (group->members()->size() + 1), errnop);
        if (!bufp) {
            *errnop = ERANGE;
            return NSS_STATUS_TRYAGAIN;
        }

        result->gr_mem = bufp;
        auto members = group->members();
        for (flatbuffers::uoffset_t i = 0; i < members->size(); ++i) {
            if (!buf.AppendString(members->Get(i)->string_view(), bufp, errnop)) {
                result->gr_mem = nullptr;
                return NSS_STATUS_TRYAGAIN;
            }
            bufp++;
        }
        *bufp = nullptr;  // End the array with a null pointer.

        return NSS_STATUS_SUCCESS;
    }

    [[nodiscard]] nss_status wrapException(const std::string& name, const std::function<nss_status()>& f) {
        try {
            return f();
        } catch (const std::exception& e) {
            SYSLOG("%s() fail: %s", name.c_str(), e.what());
        } catch (...) {
            SYSLOG("%s() fail: unknown", name.c_str());
        }

        return NSS_STATUS_UNAVAIL;
    }
} // namespace anonymous

extern "C" {

// _nss_cauth_setpwent()
// Called by NSS to open the passwd file
// 'stayopen' parameter is ignored.

enum nss_status _nss_cauth_setpwent(int stayopen) {
    Y_UNUSED(stayopen);

    return wrapException(__func__, [&] {
        passwdIter.Reset(new TUserIterator{kCachePath});
        return NSS_STATUS_SUCCESS;
    });
}

// _nss_cauth_endpwent()
// Called by NSS to close the passwd file

enum nss_status _nss_cauth_endpwent(void) {
    return wrapException(__func__, [&] {
        passwdIter->Stop();
        passwdIter.Destroy();
        return NSS_STATUS_SUCCESS;
    });

    return NSS_STATUS_SUCCESS;
}

// _nss_cauth_getpwent_r()
// Called by NSS to look up next entry in passwd file

enum nss_status _nss_cauth_getpwent_r(struct passwd* result, char* buffer,
                                      size_t buflen, int* errnop) {

    return wrapException(__func__, [&] {
        if (!passwdIter->Next()) {
            return NSS_STATUS_NOTFOUND;
        }

        auto user = passwdIter->User();
        if (!user) {
            return NSS_STATUS_NOTFOUND;
        }

        return UserForNSS(user, result, buffer, buflen, errnop);
    });
}

// _nss_cauth_getpwuid_r()
// Find a user account by uid

enum nss_status _nss_cauth_getpwuid_r(uid_t uid, struct passwd* result, char* buffer,
                                      size_t buflen, int* errnop) {

    if (uid < kMinUID) {
        return NSS_STATUS_NOTFOUND;
    }

    return wrapException(__func__, [&] {
        auto user = Singleton<TCache>(kCachePath)->UserByUid(uid);
        if (!user) {
            return NSS_STATUS_NOTFOUND;
        }

        return UserForNSS(user, result, buffer, buflen, errnop);
    });
}

// _nss_cauth_getpwnam_r()
// Find a user account by name

enum nss_status _nss_cauth_getpwnam_r(const char* name, struct passwd* result,
                                      char* buffer, size_t buflen, int* errnop) {

    if (buflen < kMinGroupBufSize) {
        *errnop = ERANGE;
        return NSS_STATUS_TRYAGAIN;
    }

    return wrapException(__func__, [&] {
        auto user = Singleton<TCache>(kCachePath)->UserByLogin(name);
        if (!user) {
            return NSS_STATUS_NOTFOUND;
        }

        return UserForNSS(user, result, buffer, buflen, errnop);
    });
}

// _nss_cauth_setgrent()
// Called by NSS to open the group file
// 'stayopen' parameter is ignored.

enum nss_status _nss_cauth_setgrent(int stayopen) {
    Y_UNUSED(stayopen);

    return wrapException(__func__, [&] {
        groupIter.Reset(new TGroupIterator{kCachePath});
        return NSS_STATUS_SUCCESS;
    });
}

// _nss_cauth_endgrent()
// Called by NSS to close the group file

enum nss_status _nss_cauth_endgrent(void) {
    return wrapException(__func__, [&] {
        groupIter->Stop();
        groupIter.Destroy();
        return NSS_STATUS_SUCCESS;
    });
}

// _nss_cauth_getgrent_r()
// Called by NSS to look up next entry in group file

enum nss_status _nss_cauth_getgrent_r(struct group* result, char* buffer,
                                      size_t buflen, int* errnop) {

    if (buflen < kMinGroupBufSize) {
        *errnop = ERANGE;
        return NSS_STATUS_TRYAGAIN;
    }

    return wrapException(__func__, [&] {
        if (!groupIter->Next()) {
            return NSS_STATUS_NOTFOUND;
        }

        auto group = groupIter->Group();
        if (!group) {
            return NSS_STATUS_NOTFOUND;
        }

        return GroupForNSS(group, result, buffer, buflen, errnop);
    });
}

// _nss_cauth_getgrgid_r()
// Find a group by gid

enum nss_status _nss_cauth_getgrgid_r(gid_t gid, struct group* result,
                                      char* buffer, size_t buflen, int* errnop) {

    if (gid < kMinGID) {
        return NSS_STATUS_NOTFOUND;
    }

    if (buflen < kMinGroupBufSize) {
        *errnop = ERANGE;
        return NSS_STATUS_TRYAGAIN;
    }

    return wrapException(__func__, [&] {
        auto group = Singleton<TCache>(kCachePath)->GroupByGid(gid);
        if (!group) {
            *errnop = ENOENT;
            return NSS_STATUS_NOTFOUND;
        }

        return GroupForNSS(group, result, buffer, buflen, errnop);
    });
}

// _nss_cauth_getgrnam_r()
// Find a group by name

enum nss_status _nss_cauth_getgrnam_r(const char* name, struct group* result,
                                      char* buffer, size_t buflen, int* errnop) {

    if (buflen < kMinGroupBufSize) {
        *errnop = ERANGE;
        return NSS_STATUS_TRYAGAIN;
    }

    return wrapException(__func__, [&] {
        auto group = Singleton<TCache>(kCachePath)->GroupByName(name);
        if (!group) {
            *errnop = ENOENT;
            return NSS_STATUS_NOTFOUND;
        }

        return GroupForNSS(group, result, buffer, buflen, errnop);
    });
}

// _nss_sss_initgroups_dyn()
// Returns a user groups

enum nss_status _nss_cauth_initgroups_dyn(const char* login, gid_t group, long int* start, long int* size,
                                          gid_t** groups, long int limit, int* errnop) {

    return wrapException(__func__, [&] {
        auto user = Singleton<TCache>(kCachePath)->UserByLogin(login);
        if (!user) {
            return NSS_STATUS_NOTFOUND;
        }

        auto userGroups = user->groups();
        ui32 numGroups = userGroups->size();
        if (numGroups > kMaxGroups) {
            numGroups = kMaxGroups;
        }

        ui32 maxRet = numGroups;
        if ((*size - *start) < numGroups) {
            ui32 newsize;
            gid_t* newgroups;

            newsize = *size + numGroups;
            if ((limit > 0) && (newsize > limit)) {
                newsize = limit;
                maxRet = newsize - *start;
            }

            newgroups = (gid_t*) std::realloc((*groups), newsize * sizeof(**groups));
            if (!newgroups) {
                *errnop = ENOMEM;
                return NSS_STATUS_TRYAGAIN;
            }

            *groups = newgroups;
            *size = newsize;
        }

        for (size_t i = 0; i < maxRet; ++i) {
            auto userGid = userGroups->Get(i);
            if (userGid <= kMinGID || userGid >= kMaxGID || userGid == group) {
                continue;
            }

            (*groups)[*start] = userGid;
            *start += 1;
        }
        return NSS_STATUS_SUCCESS;
    });
}

}
