#include "http.h"

#include <util/generic/scope.h>
#include <contrib/libs/picohttpparser/picohttpparser.h>
#include <cstdlib>
#include <cstring>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <chrono>
#include <functional>

namespace NCAuth::NSS {
    const int kSockTimeoutMs = 100;
    const struct timeval kSockTimeout = {0, kSockTimeoutMs * 1000};
#ifndef ENABLE_TEST_DEBUG
    const int kGetTimeoutMs = 300;
#endif
    const size_t kRecvBufSz = 4096;
    const char *kUserAgent = "nss_cauth_userd";

    /*
     * Send sends data of size dataSz via provided fd checking
     * if request should be cancelled calling provided cancel().
     *
     * Returns -1 in case of error (check errno for concrete error)
     *  or size of data sent via socket (must be equal to dataSz).
     * */
    ssize_t Send(int fd, const char *data, size_t dataSz, const std::function<bool()> &cancel) {
        // sendOffset tracks how many data was successfully sent via socket.
        size_t sendOffset = 0;
        while (sendOffset < dataSz) {
            // Check if request should be cancelled.
            if (cancel()) return -1;
            // data + sendOffset = beginning of pending data.
            // dataSz - sendOffset = size of pending data.
            ssize_t sret = send(fd, data + sendOffset, dataSz - sendOffset, 0);
            if (sret < 0) {
                // Check if send should be restarted.
                if (errno == EINTR) continue;
                // Restart send() in case of timeout.
                if (errno == EAGAIN || errno == EWOULDBLOCK) continue;
                else {
                    return -1;
                }
            }
            // Move beginning of pending data.
            sendOffset += sret;

            if (sendOffset < dataSz) {
                // Send more pending data.
                continue;
            } else {
                // All data was sent.
                break;
            }
        }
        return ssize_t(sendOffset);
    }

    /*
     * QueryUserd connects to unix stream socket at sockPath, issues GET request
     * to provided url and writes response data to response stream.
     *
     * Returns -1 in case of error (check errno for concrete error) or
     *  status code of request (200, 404, 500, ...).
     * */
    long QueryUserd(const char *sockPath, const TString &url, TStringStream *response) {
#ifndef ENABLE_TEST_DEBUG
        // Compute whole request deadline before any preparations.
        auto deadline = std::chrono::steady_clock::now() + std::chrono::milliseconds(kGetTimeoutMs);
#endif
        // Fill request buffer.
        TStringStream request;
        request.Reserve(512);
        request << "GET " << url << " HTTP/1.1\r\n"
                << "Host: unix\r\n"
                   "User-Agent: " << kUserAgent << "\r\n"
                << "Connection: close\r\n"
                   "Accept: */*\r\n\r\n";

        // Set up the connection.
        int fd;
        if ((fd = socket(PF_UNIX, SOCK_STREAM, 0)) < 0) return -1;
        Y_DEFER { close(fd); };
        if (setsockopt(fd, SOL_SOCKET, SO_SNDTIMEO, &kSockTimeout, sizeof(kSockTimeout)) != 0) return -1;
        if (setsockopt(fd, SOL_SOCKET, SO_RCVTIMEO, &kSockTimeout, sizeof(kSockTimeout)) != 0) return -1;
        struct sockaddr_un addr = {};
        addr.sun_family = AF_UNIX;
        strcpy(addr.sun_path, sockPath);
        if (connect(fd, (struct sockaddr *) &addr, sizeof(addr)) < 0) return -1;

        ssize_t sret = Send(fd, request.Data(), request.Size(), [&]() {
#ifndef ENABLE_TEST_DEBUG
            return deadline - std::chrono::steady_clock::now() < std::chrono::milliseconds(0);
#else
            return false;
#endif
        });
        if (sret != ssize_t(request.Size())) {
            return -1;
        }

        // Unused but needed by picohttpparser.
        int httpVer = 0;
        size_t httpMsgSz = 0;
        const char *httpMsg;

        // Set up picohttpparser useful variables.
        // Assume yandex-cauth-userd won't send many headers.
        const size_t nMaxHeaders = 16;
        struct phr_header headers[nMaxHeaders];
        int httpStatus = 0;
        size_t nHeaders = 0, // Number of headers parsed from receive buffer.
        consumedBytes = 0, // Number of data bytes written to response writer.
        dataSz = 0, // Number of data bytes should be written to response writer.
        recvOffset = 0; // recv() buffer offset in case of partial recv().
        ssize_t dataOffset = 0; // Beginning of data in buffer containing http response part.

        // Set up picohttpparser chunk decoder.
        ssize_t decodeResult = 0; // Chunk decode result.
        bool chunked = false; // Response is chunked flag.
        struct phr_chunked_decoder decoder = {};
        decoder.consume_trailer = 1;

        char *buf = (char *) malloc(kRecvBufSz);
        if (buf == nullptr) return -1;
        Y_DEFER {
                    free(buf);
                };

        ssize_t recvSz = 0, prevRecvOffset = 0;
        while ((recvSz = recv(fd, buf + recvOffset, kRecvBufSz - recvOffset, 0)) != 0) {
#ifndef ENABLE_TEST_DEBUG
            if (deadline - std::chrono::steady_clock::now() < std::chrono::milliseconds(0)) return -1;
#endif
            if (recvSz < 0) {
                if (errno == EINTR) continue;
                // retry recv() if socket timed out
                if (errno == EAGAIN || errno == EWOULDBLOCK) continue;
                return -1;
            }
            // Remember previous actual data size for phr_parse_response().
            prevRecvOffset = ssize_t(recvOffset);
            // Set recvOffset to actual size of data in buffer.
            recvOffset += recvSz;

            // Parse http response first.
            // Assume all http headers will be received in first 4k (don't know why it won't fit 4k)

            // dataOffset == 0 used as flag indicating that http response should be parsed.
            if (dataOffset == 0) {
                // nHeaders should be reset before each response parse attempt.
                nHeaders = nMaxHeaders;
                dataOffset = phr_parse_response(buf, recvOffset, &httpVer, &httpStatus, &httpMsg, &httpMsgSz,
                                                headers,
                                                &nHeaders,
                                                prevRecvOffset);
                // Check if response was partial and restart recv().
                if (dataOffset == -2) {
                    // Return error if response (not data) didn't fit buffer.
                    if (recvOffset >= kRecvBufSz) {
                        return -1;
                    }
                    // Reset dataOffset to trigger next parse attempt.
                    dataOffset = 0;
                    continue;
                } else if (dataOffset == -1) return -1;
                // Check if response is chunked.
                for (size_t i = 0; i < nHeaders; i++) {
                    // Headers are case-insensitive: https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers.
                    if (strncasecmp(headers[i].name, "Transfer-Encoding", Min(size_t(17), headers[i].name_len)) == 0) {
                        if (strncmp(headers[i].value, "chunked", Min(size_t(7), headers[i].value_len)) == 0) {
                            chunked = true;
                            break;
                        }
                    }
                }
            }
            if (!chunked) {
                // consumedBytes == 0 indicates that buffer contains http response
                if (consumedBytes == 0) {
                    dataSz = recvOffset - dataOffset;
                    response->Write(buf + dataOffset, dataSz);
                    consumedBytes += dataSz;
                } else {
                    response->Write(buf, recvSz);
                    consumedBytes += recvSz;
                }
            } else {
                // consumedBytes == 0 indicates that buffer contains http response
                if (consumedBytes == 0) {
                    dataSz = recvOffset - dataOffset;
                    // Decode part of chunked response starting from dataOffset.
                    decodeResult = phr_decode_chunked(&decoder, buf + dataOffset, &dataSz);
                    if (decodeResult == -1) return -1;
                    response->Write(buf + dataOffset, dataSz);
                    consumedBytes += dataSz;
                } else {
                    // Decode part of chunked response.
                    decodeResult = phr_decode_chunked(&decoder, buf, (size_t*)&recvSz);
                    if (decodeResult == -1) return -1;
                    response->Write(buf, recvSz);
                    consumedBytes += recvSz;
                }
            }
            // Reset recvOffset if response data was successfully written to response writer.
            recvOffset = 0;
        }
        return httpStatus;
    }
}
