#include <grp.h>
#include <nss.h>
#include <pwd.h>
#include <syslog.h>
#include <cstdlib>

#include <util/stream/str.h>
#include <util/string/builder.h>
#include <library/cpp/uri/encode.h>

#include <infra/cauth/agent/linux/nss_cauth_userd/http/http.h>
#include <infra/cauth/agent/linux/nss_utils/utils.h>
#include <infra/cauth/agent/linux/yandex-cauth-userd/pkg/cauthrpc/models.pb.h>


using namespace NCAuth::NSS;

#define SYSLOG(fmt, ...)                                          \
  do {                                                            \
      openlog("nss_cauth_userd", LOG_PID|LOG_PERROR, LOG_USER);   \
      syslog(LOG_ERR, fmt, ##__VA_ARGS__);                        \
      closelog();                                                 \
  } while (0)


namespace {
    // Arcadia uses gid 15513 override for all users, see CAUTH-2593 for details.
    // Limit min uid/gid  greater than 10000 instead of previous 20000 value.
    const __uid_t kMinUID = 10000;
    const __gid_t kMinGID = 10000;
    const __uid_t kMaxUID = (uid_t) -2;
    const __gid_t kMaxGID = (gid_t) -2;
    const ui32 kMaxUserGroups = 1000;
    const size_t kMinGroupBufSize = 4096;
    const size_t kMinHttpBufSize = 1024;

    const char* kBaseURI = "/nss/v1";
    const char *kApiPath = "/run/yandex-cauth-userd.sock";

    nss_status UserForNSS(const ::NCAuth::NCAuthAgent::NRpc::User& user, struct passwd* result,
                          char* buffer, size_t buflen, int* errnop) {

        if (user.uid() <= kMinUID || user.uid() >= kMaxUID || user.gid() <= kMinGID || user.gid() >= kMaxGID) {
            return NSS_STATUS_UNAVAIL;
        }

        result->pw_uid = user.uid();
        result->pw_gid = user.gid();
        result->pw_passwd = (char*) "*";

        NUtils::TBufferManager buf(buffer, buflen);
        if (!buf.AppendString(user.login(), &result->pw_name, errnop)) {
            return NSS_STATUS_TRYAGAIN;
        }

        if (!buf.AppendString(user.gecos(), &result->pw_gecos, errnop)) {
            return NSS_STATUS_TRYAGAIN;
        }

        if (!buf.AppendString(user.home_dir(), &result->pw_dir, errnop)) {
            return NSS_STATUS_TRYAGAIN;
        }

        if (!buf.AppendString(user.shell(), &result->pw_shell, errnop)) {
            return NSS_STATUS_TRYAGAIN;
        }

        return NSS_STATUS_SUCCESS;
    }

    nss_status GroupForNSS(const ::NCAuth::NCAuthAgent::NRpc::Group& group, struct group* result,
                           char* buffer, size_t buflen, int* errnop) {


        if (group.gid() <= kMinGID || group.gid() >= kMaxGID) {
            return NSS_STATUS_UNAVAIL;
        }

        result->gr_gid = group.gid();
        result->gr_passwd = (char*) "*";

        NUtils::TBufferManager buf(buffer, buflen);

        if (!buf.AppendString(group.name(), &result->gr_name, errnop)) {
            return NSS_STATUS_TRYAGAIN;
        }

        char** bufp;
        bufp = (char**) buf.Reserve(sizeof(char*) * (group.membersSize() + 1), errnop);
        if (!bufp) {
            *errnop = ERANGE;
            return NSS_STATUS_TRYAGAIN;
        }

        result->gr_mem = bufp;

        for (auto&& member: group.members()) {
            if (!buf.AppendString(member, bufp, errnop)) {
                result->gr_mem = nullptr;
                return NSS_STATUS_TRYAGAIN;
            }
            bufp++;
        }
        *bufp = nullptr;  // End the array with a null pointer.

        return NSS_STATUS_SUCCESS;
    }

    [[nodiscard]] nss_status wrapException(const TString& name, const std::function<nss_status()>& f) {
        try {
            return f();
        } catch (const std::exception& e) {
            SYSLOG("%s() fail: %s", name.c_str(), e.what());
        } catch (...) {
            SYSLOG("%s() fail: unknown", name.c_str());
        }

        return NSS_STATUS_UNAVAIL;
    }

    [[nodiscard]] nss_status callCauth(const TString& url, TStringStream* out) {
        out->Reserve(kMinHttpBufSize);
        auto status = QueryUserd(kApiPath, url, out);
        switch (status) {
            case 404:
                return NSS_STATUS_NOTFOUND;
            case 200:
                return NSS_STATUS_SUCCESS;
            default:
                return NSS_STATUS_UNAVAIL;
        };
    }

} // namespace anonymous

extern "C" {

// _nss_cauth_userd_setpwent()
// Called by NSS to open the passwd file
// we don't allow to iterate throught users for now.

enum nss_status _nss_cauth_userd_setpwent(int stayopen) {
    Y_UNUSED(stayopen);

    return NSS_STATUS_UNAVAIL;
}

// _nss_cauth_userd_endpwent()
// Called by NSS to close the passwd file

enum nss_status _nss_cauth_userd_endpwent(void) {
    return NSS_STATUS_UNAVAIL;
}

// _nss_cauth_userd_getpwent_r()
// Called by NSS to look up next entry in passwd file

enum nss_status _nss_cauth_userd_getpwent_r(struct passwd* result, char* buffer,
                                            size_t buflen, int* errnop) {

    Y_UNUSED(result, buffer, buflen, errnop);
    return NSS_STATUS_NOTFOUND;
}

// _nss_cauth_userd_getpwuid_r()
// Find a user account by uid

enum nss_status _nss_cauth_userd_getpwuid_r(uid_t uid, struct passwd* result, char* buffer,
                                            size_t buflen, int* errnop) {

    if (uid <= kMinUID) {
        return NSS_STATUS_NOTFOUND;
    }

    return wrapException(__func__, [&] {
        TStringStream url;
        url << kBaseURI << "/user/" << uid;

        TStringStream rsp;
        nss_status ret = callCauth(url.Str(), &rsp);
        if (ret != NSS_STATUS_SUCCESS) {
            return ret;
        }

        ::NCAuth::NCAuthAgent::NRpc::User user;
        if (!user.ParseFromString(rsp.Str())) {
            return NSS_STATUS_UNAVAIL;
        }

        return UserForNSS(user, result, buffer, buflen, errnop);
    });
}

// _nss_cauth_userd_getpwnam_r()
// Find a user account by name

enum nss_status _nss_cauth_userd_getpwnam_r(const char* login, struct passwd* result,
                                            char* buffer, size_t buflen, int* errnop) {

    return wrapException(__func__, [&] {
        TStringStream url;
        url << kBaseURI << "/user/";
        NUri::TEncoder::EncodeField(url, login, NUri::TField::EField::FieldPath);

        TStringStream rsp;
        nss_status ret = callCauth(url.Str(), &rsp);
        if (ret != NSS_STATUS_SUCCESS) {
            return ret;
        }

        ::NCAuth::NCAuthAgent::NRpc::User user;
        if (!user.ParseFromString(rsp.Str())) {
            return NSS_STATUS_UNAVAIL;
        }

        return UserForNSS(user, result, buffer, buflen, errnop);
    });
}

// _nss_cauth_userd_setgrent()
// Called by NSS to open the group file
// 'stayopen' parameter is ignored.

enum nss_status _nss_cauth_userd_setgrent(int stayopen) {
    Y_UNUSED(stayopen);

    return NSS_STATUS_UNAVAIL;
}

// _nss_cauth_userd_endgrent()
// Called by NSS to close the group file

enum nss_status _nss_cauth_userd_endgrent(void) {

    return NSS_STATUS_SUCCESS;
}

// _nss_cauth_userd_getgrent_r()
// Called by NSS to look up next entry in group file

enum nss_status _nss_cauth_userd_getgrent_r(struct group* result, char* buffer,
                                            size_t buflen, int* errnop) {

    Y_UNUSED(result, buffer, buflen, errnop);
    return NSS_STATUS_NOTFOUND;
}

// _nss_cauth_userd_getgrgid_r()
// Find a group by gid

enum nss_status _nss_cauth_userd_getgrgid_r(gid_t gid, struct group* result,
                                            char* buffer, size_t buflen, int* errnop) {

    if (gid < kMinGID) {
        return NSS_STATUS_NOTFOUND;
    }

    if (buflen < kMinGroupBufSize) {
        *errnop = ERANGE;
        return NSS_STATUS_TRYAGAIN;
    }

    return wrapException(__func__, [&] {
        TStringStream url;
        url << kBaseURI << "/group/" << gid;

        TStringStream rsp;
        nss_status ret = callCauth(url.Str(), &rsp);
        if (ret != NSS_STATUS_SUCCESS) {
            return ret;
        }

        ::NCAuth::NCAuthAgent::NRpc::Group group;
        if (!group.ParseFromString(rsp.Str())) {
            return NSS_STATUS_UNAVAIL;
        }

        return GroupForNSS(group, result, buffer, buflen, errnop);
    });
}

// _nss_cauth_userd_getgrnam_r()
// Find a group by name

enum nss_status _nss_cauth_userd_getgrnam_r(const char* name, struct group* result,
                                            char* buffer, size_t buflen, int* errnop) {

    if (buflen < kMinGroupBufSize) {
        *errnop = ERANGE;
        return NSS_STATUS_TRYAGAIN;
    }

    return wrapException(__func__, [&] {
        TStringStream url;
        url << kBaseURI << "/group/";
        NUri::TEncoder::EncodeField(url, name, NUri::TField::EField::FieldPath);

        TStringStream rsp;
        nss_status ret = callCauth(url.Str(), &rsp);
        if (ret != NSS_STATUS_SUCCESS) {
            return ret;
        }

        ::NCAuth::NCAuthAgent::NRpc::Group group;
        if (!group.ParseFromString(rsp.Str())) {
            return NSS_STATUS_UNAVAIL;
        }

        return GroupForNSS(group, result, buffer, buflen, errnop);
    });
}

// _nss_sss_initgroups_dyn()
// Returns a user groups

enum nss_status _nss_cauth_userd_initgroups_dyn(const char* login, gid_t group, long int* start, long int* size,
                                                gid_t** groups, long int limit, int* errnop) {

    return wrapException(__func__, [&] {
        TStringStream url;
        url << kBaseURI << "/groups/";
        NUri::TEncoder::EncodeField(url, login, NUri::TField::EField::FieldPath);

        TStringStream rsp;
        nss_status ret = callCauth(url.Str(), &rsp);
        if (ret != NSS_STATUS_SUCCESS) {
            return ret;
        }

        ::NCAuth::NCAuthAgent::NRpc::UserGroups userGroups;
        if (!userGroups.ParseFromString(rsp.Str())) {
            return NSS_STATUS_UNAVAIL;
        }

        ui32 numGroups = userGroups.groups().size();
        if (numGroups > kMaxUserGroups) {
            numGroups = kMaxUserGroups;
        }

        ui32 maxRet = numGroups;
        if ((*size - *start) < numGroups) {
            ui32 newsize;
            gid_t* newgroups;

            newsize = *size + numGroups;
            if ((limit > 0) && (newsize > limit)) {
                newsize = limit;
                maxRet = newsize - *start;
            }

            newgroups = (gid_t*) std::realloc((*groups), newsize * sizeof(**groups));
            if (!newgroups) {
                *errnop = ENOMEM;
                return NSS_STATUS_TRYAGAIN;
            }

            *groups = newgroups;
            *size = newsize;
        }

        for (size_t i = 0; i < maxRet; ++i) {
            auto userGid = userGroups.groups().Get(i);
            if (userGid <= kMinGID || userGid >= kMaxGID || userGid == group) {
                continue;
            }

            (*groups)[*start] = userGid;
            *start += 1;
        }
        return NSS_STATUS_SUCCESS;
    });
}

}
