package main

import (
	"context"
	"sort"

	flatbuffers "github.com/google/flatbuffers/go"

	"a.yandex-team.ru/infra/cauth/agent/linux/flatcache/go/flatcache"
	"a.yandex-team.ru/infra/cauth/agent/linux/nss_export/provider"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/slices"
)

var excludedGroups = []string{
	"hw-watcher",
	"monitor",
	"root",
	"skynet",
	"skywheel",
	"consul",
	"dpt_yandex",
	"fired",
	"dpt_outstaff",
	"newbie",
}

type Groups struct {
	GroupsOffset flatbuffers.UOffsetT
	NamesOffset  flatbuffers.UOffsetT
	UserGroups   map[string][]uint32
}

func getGroups(p provider.Provider, b *flatbuffers.Builder) (Groups, error) {
	entries, err := p.Groups(context.TODO())
	if err != nil {
		return Groups{}, err
	}

	groups := make([]uidedOffset, 0, len(entries))
	groupNames := make([]namedOffset, 0, len(entries))
	userGroups := make(map[string][]uint32)
	for _, entry := range entries {
		if slices.ContainsString(excludedGroups, entry.Name) {
			logger.Info("skip excluded group",
				log.String("group_name", entry.Name))
			continue
		}

		memberNames := make([]flatbuffers.UOffsetT, 0, len(entry.Members))
		for _, member := range entry.Members {
			userGroups[member] = append(userGroups[member], entry.GID)
			memberNames = append(memberNames, b.CreateSharedString(member))
		}

		name := b.CreateSharedString(entry.Name)
		flatcache.GroupStartMembersVector(b, len(memberNames))
		for i := len(memberNames) - 1; i >= 0; i-- {
			b.PrependUOffsetT(memberNames[i])
		}
		members := b.EndVector(len(memberNames))

		flatcache.GroupStart(b)
		flatcache.GroupAddGid(b, entry.GID)
		flatcache.GroupAddName(b, name)
		flatcache.GroupAddMembers(b, members)
		flatGroup := flatcache.GroupEnd(b)
		groups = append(groups, uidedOffset{
			uid:    entry.GID,
			offset: flatGroup,
		})

		flatcache.GroupNameStart(b)
		flatcache.GroupNameAddName(b, name)
		flatcache.GroupNameAddGroup(b, flatGroup)
		groupNames = append(groupNames, namedOffset{
			name:   entry.Name,
			offset: flatcache.UserNameEnd(b),
		})
	}

	sort.Slice(groups, func(i, j int) bool {
		return groups[i].uid < groups[j].uid
	})

	sort.Slice(groupNames, func(i, j int) bool {
		return groupNames[i].name < groupNames[j].name
	})

	numItems := len(groups)
	flatcache.CacheStartGroupsVector(b, numItems)
	for i := numItems - 1; i >= 0; i-- {
		b.PrependUOffsetT(groups[i].offset)
	}
	groupsOffset := b.EndVector(numItems)

	flatcache.CacheStartGroupNamesVector(b, numItems)
	for i := numItems - 1; i >= 0; i-- {
		b.PrependUOffsetT(groupNames[i].offset)
	}
	namesOffset := b.EndVector(numItems)

	return Groups{
		GroupsOffset: groupsOffset,
		NamesOffset:  namesOffset,
		UserGroups:   userGroups,
	}, nil
}
