package cauth

import (
	"context"
	"crypto/tls"
	"fmt"
	"strconv"
	"strings"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/infra/cauth/agent/linux/nss_export/provider"
	"a.yandex-team.ru/library/go/certifi"
)

var _ provider.Provider = (*Provider)(nil)

type Provider struct {
	httpc *resty.Client
}

func NewProvider() *Provider {
	httpc := resty.New().
		SetHostURL("https://cauth.yandex.net:4443").
		SetRetryCount(5).
		SetDoNotParseResponse(true).
		SetRedirectPolicy(resty.NoRedirectPolicy())

	certPool, err := certifi.NewCertPool()
	if err == nil {
		httpc.SetTLSClientConfig(&tls.Config{RootCAs: certPool})
	}

	return &Provider{
		httpc: httpc,
	}
}

func (p *Provider) Users(ctx context.Context) ([]provider.User, error) {
	rsp, err := p.httpc.R().SetContext(ctx).Get("/passwd/serverusers")
	if err != nil {
		return nil, fmt.Errorf("can't get cauth passwd: %w", err)
	}

	body := rsp.RawBody()
	defer func() { _ = body.Close() }()

	var users []provider.User
	err = ReadColonFile(body, func(line []byte) error {
		// alonger:*:31592:87443:alonger:/home/alonger:/bin/bash
		parts := strings.SplitN(string(line), ":", 7)
		if len(parts) < 7 || parts[0] == "" || parts[0][0] == '+' || parts[0][0] == '-' {
			return nil
		}

		uid, err := strconv.ParseUint(parts[2], 10, 32)
		if err != nil {
			return err
		}

		gid, err := strconv.ParseUint(parts[3], 10, 32)
		if err != nil {
			return err
		}

		users = append(users, provider.User{
			UID:     uint32(uid),
			GID:     uint32(gid),
			Login:   parts[0],
			HomeDir: parts[5],
			Shell:   parts[6],
		})
		return nil
	})

	if err != nil {
		return nil, fmt.Errorf("parse failed: %w", err)
	}

	return users, nil
}

func (p *Provider) Groups(ctx context.Context) ([]provider.Group, error) {
	rsp, err := p.httpc.R().SetContext(ctx).Get("/group/serverusers")
	if err != nil {
		return nil, fmt.Errorf("can't get cauth groups: %w", err)
	}

	body := rsp.RawBody()
	defer func() { _ = body.Close() }()

	var groups []provider.Group
	err = ReadColonFile(body, func(line []byte) error {
		// svc_hostmanager-walle_development:x:237391:mixas,warwish
		parts := strings.SplitN(string(line), ":", 4)
		if len(parts) < 4 || parts[0] == "" || parts[0][0] == '+' || parts[0][0] == '-' {
			return nil
		}
		gid, err := strconv.ParseUint(parts[2], 10, 32)
		if err != nil {
			return err
		}

		members := strings.Split(parts[3], ",")
		for i, member := range members {
			members[i] = strings.TrimSpace(member)
		}

		groups = append(groups, provider.Group{
			GID:     uint32(gid),
			Name:    parts[0],
			Members: members,
		})
		return nil
	})

	if err != nil {
		return nil, fmt.Errorf("parse failed: %w", err)
	}

	return groups, nil
}
