package ldap

import (
	"context"
	"fmt"
	"strconv"

	"gopkg.in/ldap.v3"

	"a.yandex-team.ru/infra/cauth/agent/linux/nss_export/provider"
)

var _ provider.Provider = (*Provider)(nil)

type Provider struct {
}

func NewProvider() *Provider {
	return &Provider{}
}

func (p *Provider) Users(_ context.Context) ([]provider.User, error) {
	conn, err := p.dial()
	if err != nil {
		return nil, err
	}
	defer conn.Close()

	res, err := conn.Search(&ldap.SearchRequest{
		BaseDN: "ou=people,dc=yandex,dc=net",
		Filter: "(&(objectClass=posixAccount)(!(gidNumber:=20000)))",
		Scope:  ldap.ScopeWholeSubtree,
		Attributes: []string{
			"cn",
			"homeDirectory",
			"loginShell",
			"uid",
			"uidNumber",
			"gidNumber",
		},
	})

	if err != nil {
		return nil, fmt.Errorf("ldap search fail: %w", err)
	}

	users := make([]provider.User, 0, len(res.Entries))
	for _, entry := range res.Entries {
		uid, err := strconv.ParseUint(entry.GetAttributeValue("uidNumber"), 10, 32)
		if err != nil {
			return nil, fmt.Errorf("can't parse uidNumber (dn=%q): %w", entry.DN, err)
		}

		gid, err := strconv.ParseUint(entry.GetAttributeValue("gidNumber"), 10, 32)
		if err != nil {
			return nil, fmt.Errorf("can't parse gidNumber (dn=%q): %w", entry.DN, err)
		}

		users = append(users, provider.User{
			UID:     uint32(uid),
			GID:     uint32(gid),
			Login:   entry.GetAttributeValue("uid"),
			HomeDir: entry.GetAttributeValue("homeDirectory"),
			Shell:   entry.GetAttributeValue("loginShell"),
		})
	}

	return users, nil
}

func (p *Provider) Groups(_ context.Context) ([]provider.Group, error) {
	conn, err := p.dial()
	if err != nil {
		return nil, err
	}
	defer conn.Close()

	res, err := conn.Search(&ldap.SearchRequest{
		BaseDN: "ou=groups,dc=yandex,dc=net",
		Filter: "(objectClass=posixGroup)",
		Scope:  ldap.ScopeWholeSubtree,
		Attributes: []string{
			"cn",
			"gidNumber",
			"memberUid",
		},
	})
	if err != nil {
		return nil, fmt.Errorf("ldap search fail: %w", err)
	}

	groups := make([]provider.Group, 0, len(res.Entries))
	for _, entry := range res.Entries {
		gid, err := strconv.ParseUint(entry.GetAttributeValue("gidNumber"), 10, 32)
		if err != nil {
			return nil, fmt.Errorf("can't parse gidNumber (dn=%q): %w", entry.DN, err)
		}

		groups = append(groups, provider.Group{
			GID:     uint32(gid),
			Name:    entry.GetAttributeValue("cn"),
			Members: entry.GetAttributeValues("memberUid"),
		})
	}

	return groups, nil
}

func (p *Provider) dial() (*ldap.Conn, error) {
	l, err := ldap.Dial("tcp", "cauth6.yandex.net:389")
	if err != nil {
		return nil, err
	}

	_, err = l.SimpleBind(&ldap.SimpleBindRequest{
		AllowEmptyPassword: true,
	})
	if err != nil {
		return nil, err
	}

	return l, nil
}
