package main

import (
	"context"
	"sort"

	flatbuffers "github.com/google/flatbuffers/go"

	"a.yandex-team.ru/infra/cauth/agent/linux/flatcache/go/flatcache"
	"a.yandex-team.ru/infra/cauth/agent/linux/nss_export/provider"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/slices"
)

var excludedUsers = []string{
	"avahi-autoipd",
	"backup",
	"bin",
	"daemon",
	"hw-watcher",
	"loadbase",
	"mail",
	"messagebus",
	"monitor",
	"nobody",
	"ntp",
	"postfix",
	"root",
	"skynet",
	"sshd",
	"sys",
	"syslog",
	"ubuntu",
	"www-data",
	"consul",
}

type Users struct {
	UsersOffset flatbuffers.UOffsetT
	NamesOffset flatbuffers.UOffsetT
}

func getUsers(p provider.Provider, b *flatbuffers.Builder, groups Groups) (Users, error) {
	entries, err := p.Users(context.TODO())
	if err != nil {
		return Users{}, err
	}

	users := make([]uidedOffset, 0, len(entries))
	userNames := make([]namedOffset, 0, len(entries))
	for _, entry := range entries {
		if slices.ContainsString(excludedUsers, entry.Login) {
			logger.Info("skip excluded user",
				log.String("login", entry.Login))
			continue
		}

		login := b.CreateSharedString(entry.Login)
		homeDir := b.CreateString(entry.HomeDir)
		shell := b.CreateSharedString(entry.Shell)
		userGroups := groups.UserGroups[entry.Login]

		flatcache.UserStartGroupsVector(b, len(userGroups))
		numGroups := 0
		for i := len(userGroups) - 1; i >= 0; i-- {
			secondaryGID := userGroups[i]
			if entry.GID == secondaryGID {
				continue
			}

			numGroups++
			b.PrependUint32(secondaryGID)
		}

		groupsOffset := b.EndVector(numGroups)

		flatcache.UserStart(b)
		flatcache.UserAddUid(b, entry.UID)
		flatcache.UserAddGid(b, entry.GID)
		flatcache.UserAddLogin(b, login)
		flatcache.UserAddHomeDir(b, homeDir)
		flatcache.UserAddShell(b, shell)
		flatcache.UserAddGroups(b, groupsOffset)
		flatUser := flatcache.UserEnd(b)
		users = append(users, uidedOffset{
			uid:    entry.UID,
			offset: flatUser,
		})

		flatcache.UserNameStart(b)
		flatcache.UserNameAddLogin(b, login)
		flatcache.UserNameAddUser(b, flatUser)
		userNames = append(userNames, namedOffset{
			name:   entry.Login,
			offset: flatcache.UserNameEnd(b),
		})
	}

	sort.Slice(users, func(i, j int) bool {
		return users[i].uid < users[j].uid
	})

	sort.Slice(userNames, func(i, j int) bool {
		return userNames[i].name < userNames[j].name
	})

	numItems := len(users)
	flatcache.CacheStartUsersVector(b, numItems)
	for i := numItems - 1; i >= 0; i-- {
		b.PrependUOffsetT(users[i].offset)
	}
	usersOffset := b.EndVector(numItems)

	flatcache.CacheStartUsersVector(b, numItems)
	for i := numItems - 1; i >= 0; i-- {
		b.PrependUOffsetT(userNames[i].offset)
	}
	namesOffset := b.EndVector(numItems)

	return Users{
		UsersOffset: usersOffset,
		NamesOffset: namesOffset,
	}, nil
}
