package app

import (
	"context"
	"fmt"
	"net"
	"net/http"
	"os"
	"strconv"

	"a.yandex-team.ru/infra/cauth/agent/linux/yandex-cauth-userd/internal/cauth"
	"a.yandex-team.ru/infra/cauth/agent/linux/yandex-cauth-userd/internal/config"
	"a.yandex-team.ru/infra/cauth/agent/linux/yandex-cauth-userd/internal/storage"
	"a.yandex-team.ru/infra/hmserver/pkg/yasmclient"
	"a.yandex-team.ru/library/go/core/log"
)

const (
	envFDsName     = "LISTEN_FDS"
	envPIDName     = "LISTEN_PID"
	envFDNamesName = "LISTEN_FDNAMES"
)

type App struct {
	cfg     *config.Config
	http    *http.Server
	log     log.Logger
	stor    storage.Controller
	cauth   cauth.Client
	repo    storage.Repo
	metrics *metrics
	yasm    *yasmclient.YasmClient
}

func NewApp(cfg *config.Config, l log.Logger) (app *App, err error) {
	repo, err := storage.NewTimestampRepo(&cfg.RepoConfig)
	if err != nil {
		return nil, err
	}
	yasmTags := map[string]string{"itype": cfg.YASMConfig.IType, "ctype": cfg.YASMConfig.CType}
	userOverrides := &storage.UserOverrides{
		VirtualUsersShellOverride: cfg.DaemonConfig.VirtualUsersShellOverride,
		VirtualUsersGIDOverride:   cfg.DaemonConfig.VirtualUsersGIDOverride,
		RealUsersGIDOverride:      cfg.DaemonConfig.RealUsersGIDOverride,
		RealUsersShellOverride:    cfg.DaemonConfig.RealUsersShellOverride,
	}
	app = &App{
		cfg:   cfg,
		log:   l,
		cauth: cauth.NewClient(&cfg.CAuthConfig),
		repo:  repo,
		stor: storage.NewController(cfg.DaemonConfig.MaxUpdatesPerMinute,
			cfg.DaemonConfig.UpdateHistoryFile, cfg.DaemonConfig.ResolveGroupMembers, userOverrides),
		metrics: newMetrics(yasmTags),
		yasm:    yasmclient.NewLocalWithURL(cfg.YASMConfig.YASMURL),
	}

	app.http = &http.Server{
		Handler: newRouter(app),
	}

	return app, nil
}

func (a *App) Start() error {
	a.log.Info("start http API", log.String("addr", a.cfg.DaemonConfig.SocketPath))
	var listener net.Listener
	var err error

	if a.cfg.DaemonConfig.EnableSocketActivation {
		// SocketFromSystemd checks if we have socket provided by systemd according to:
		//  * https://www.freedesktop.org/software/systemd/man/systemd.socket.html - we expect stream non-blocking socket
		//  * http://0pointer.net/blog/projects/socket-activated-containers.html - more motivation and simply a good reading from the author

		envPID := os.Getenv(envPIDName)
		envFDs := os.Getenv(envFDsName)
		pid, err := strconv.Atoi(envPID)
		if err != nil || pid != os.Getpid() {
			return fmt.Errorf("socket activation failed: %s value %s is not equal pid value", envPIDName, envPID)
		}
		fds, err := strconv.Atoi(envFDs)
		if err != nil || fds != 1 {
			return fmt.Errorf("socket activation failed: invalid %s has value %s", envFDsName, envFDs)
		}
		fileName := os.Getenv(envFDNamesName)
		if fileName == "" {
			fileName = "LISTEN_FD_3"
		}
		listener, err = net.FileListener(os.NewFile(3, fileName))
		if err != nil {
			return err
		}

	} else {
		_ = os.Remove(a.cfg.DaemonConfig.SocketPath)

		listener, err = net.Listen("unix", a.cfg.DaemonConfig.SocketPath)
		if err != nil {
			return err
		}
		_ = os.Chmod(a.cfg.DaemonConfig.SocketPath, 0o666)
	}
	defer func() {
		_ = listener.Close()
	}()
	if !a.stor.HasIndex(a.repo) {
		err = a.stor.UpdateIndex(context.Background(), a.cauth, a.repo)
	} else {
		err = a.stor.LoadIndex(context.Background(), a.repo)
	}
	if err != nil {
		return err
	}
	return a.http.Serve(listener)
}

func (a *App) Reload() error {
	// TODO: reload config here?
	return a.stor.UpdateIndex(context.Background(), a.cauth, a.repo)
}

func (a *App) Shutdown(ctx context.Context) error {
	return a.http.Shutdown(ctx)
}

func (a *App) PushMetrics(ctx context.Context) error {
	return a.yasm.SendMetrics(ctx, a.metrics.yasmPushSignals())
}
