package app

import (
	"math"
	"sync"
	"time"

	hmm "a.yandex-team.ru/infra/hmserver/pkg/metrics"
	"a.yandex-team.ru/infra/hmserver/pkg/yasmclient"
)

const (
	numBuckets                = 20
	metricRequestsTimingsName = "cauth_userd_requests_time_hgram"
	metricRequests2xxName     = "cauth_userd_requests_2xx_summ"
	metricRequests4xxName     = "cauth_userd_requests_4xx_summ"
	metricRequests5xxName     = "cauth_userd_requests_2xx_summ"
	yasmTTLSec                = 60
)

var timingsBuckets = initBuckets(numBuckets)

type metrics struct {
	mu                sync.Mutex
	requestsTimingsUs *hmm.Histogram
	requests2xx       *hmm.Gauge
	requests4xx       *hmm.Gauge
	requests5xx       *hmm.Gauge
	tags              map[string]string
}

func newMetrics(tags map[string]string) *metrics {
	return &metrics{
		mu:                sync.Mutex{},
		requestsTimingsUs: hmm.NewHistogram(metricRequestsTimingsName, timingsBuckets),
		requests2xx:       hmm.NewGauge(metricRequests2xxName),
		requests4xx:       hmm.NewGauge(metricRequests4xxName),
		requests5xx:       hmm.NewGauge(metricRequests5xxName),
		tags:              tags,
	}
}

func (m *metrics) PutRequestDuration(d time.Duration) {
	m.mu.Lock()
	defer m.mu.Unlock()
	m.requestsTimingsUs.Observe(float64(d.Microseconds()))
}

func (m *metrics) PutRequestStatus(code int) {
	m.mu.Lock()
	defer m.mu.Unlock()
	if code >= 200 && code < 300 {
		m.requests2xx.Value += 1
		return
	}
	if code >= 400 && code < 500 {
		m.requests4xx.Value += 1
		return
	}
	if code >= 500 {
		m.requests5xx.Value += 1
	}
}

func (m *metrics) yasmPushSignals() []yasmclient.YasmMetrics {
	m.mu.Lock()
	defer m.mu.Unlock()
	rv := []yasmclient.YasmMetrics{
		{
			Tags:   m.tags,
			TTL:    yasmTTLSec,
			Values: []yasmclient.YasmValue{m.requestsTimingsUs.FmtPush()},
		},
		{
			Tags:   m.tags,
			Values: []yasmclient.YasmValue{m.requests2xx.FmtPush(), m.requests4xx.FmtPush(), m.requests5xx.FmtPush()},
		},
	}
	// Reset gauge values
	m.requests2xx.Set(0)
	m.requests4xx.Set(0)
	m.requests5xx.Set(0)
	return rv
}

func initBuckets(n int) []float64 {
	rv := make([]float64, n)
	for i := 0; i < n; i++ {
		rv[i] = math.Pow(1.5, float64(i))
	}
	return rv
}
