package cauth

import (
	"bytes"
	"context"
	"crypto/tls"
	"fmt"
	"io"
	"net/url"

	"a.yandex-team.ru/infra/cauth/agent/linux/yandex-cauth-userd/internal/config"
	"a.yandex-team.ru/library/go/certifi"
	"github.com/go-resty/resty/v2"
	"github.com/klauspost/compress/zstd"
)

// PasswdFetcher defines interface to fetch passwd (user list data)
type PasswdFetcher interface {
	FetchPasswd(ctx context.Context) (io.ReadCloser, error)
}

// GroupFetcher defines interface to fetch group (group list data)
type GroupFetcher interface {
	FetchGroup(ctx context.Context) (io.ReadCloser, error)
}

// AccessFetcher defines interface to fetch yandex-access-custom.conf data
type AccessFetcher interface {
	FetchAccess(ctx context.Context) (io.ReadCloser, error)
}

// AdminsUsersFetcher defines interface to fetch serveradmins passwd
type AdminsUsersFetcher interface {
	FetchAdminUsers(ctx context.Context) (io.ReadCloser, error)
}

// KeysFetcher defines interface to fetch users keys
type KeysFetcher interface {
	FetchKeys(ctx context.Context) (io.ReadCloser, error)
}

// SudoersFetcher defines interface to fetch sudoers
type SudoersFetcher interface {
	FetchSudoers(ctx context.Context) (io.ReadCloser, error)
}

// KeysInfoFetcher defines interface to fetch Keys Info
type KeysInfoFetcher interface {
	FetchKeysInfo(ctx context.Context) (io.ReadCloser, error)
}

// InsecureFetcher defines interface to fetch insecure CA
type InsecureFetcher interface {
	FetchInsecure(ctx context.Context) (io.ReadCloser, error)
}

// SecureFetcher defines interface to fetch secure CA
type SecureFetcher interface {
	FetchSecure(ctx context.Context) (io.ReadCloser, error)
}

// SudoFetcher defines interface to fetch sudo CA
type SudoFetcher interface {
	FetchSudo(ctx context.Context) (io.ReadCloser, error)
}

// KRLFetcher defines interface to fetch krl
type KRLFetcher interface {
	FetchKRL(ctx context.Context) (io.ReadCloser, error)
}

// Client interface defines composition of required fetchers
type Client interface {
	PasswdFetcher
	GroupFetcher
	AccessFetcher
	KeysFetcher
	AdminsUsersFetcher
	SudoersFetcher
	KeysInfoFetcher
	InsecureFetcher
	SecureFetcher
	SudoFetcher
	KRLFetcher
}

type client struct {
	c          *resty.Client
	keysClient *resty.Client
	keysInfo   *KeysInfo
}

func (c *client) requestForURL(URL string) (*resty.Request, error) {
	// Determine if we can use shared between requests to cauth API client
	// or we should use raw keysClient for issuing requests.
	req := c.c.R()
	u, parseErr := url.Parse(URL)
	if parseErr != nil {
		return nil, fmt.Errorf("failed to parse url %s: %w", URL, parseErr)
	}
	if u.Host != "" {
		req = c.keysClient.R()
	}
	return req, nil
}

func (c *client) get(ctx context.Context, URL string) (io.ReadCloser, error) {
	req, err := c.requestForURL(URL)
	if err != nil {
		return nil, err
	}
	rsp, err := req.SetContext(ctx).Get(URL)
	if err != nil {
		return nil, fmt.Errorf("failed to fetch %s: %w", req.URL, err)
	}

	if rsp.IsSuccess() && rsp.StatusCode() == 200 {
		return rsp.RawBody(), nil
	}

	return nil, fmt.Errorf("unexpected status code while fetching %s: %d", req.URL, rsp.StatusCode())
}

func (c *client) FetchPasswd(ctx context.Context) (io.ReadCloser, error) {
	return c.get(ctx, "/passwd/serverusers/")
}

func (c *client) FetchGroup(ctx context.Context) (io.ReadCloser, error) {
	return c.get(ctx, "/group/serverusers/")
}

func (c *client) FetchAccess(ctx context.Context) (io.ReadCloser, error) {
	return c.get(ctx, "/access/")
}

func (c *client) FetchKeys(ctx context.Context) (io.ReadCloser, error) {
	return c.get(ctx, "/userkeys/")
}

func (c *client) FetchAdminUsers(ctx context.Context) (io.ReadCloser, error) {
	return c.get(ctx, "/passwd/serveradmins/")
}

func (c *client) FetchSudoers(ctx context.Context) (io.ReadCloser, error) {
	return c.get(ctx, "/sudoers/")
}

func (c *client) FetchKeysInfo(ctx context.Context) (io.ReadCloser, error) {
	result, err := c.get(ctx, "/keysinfo/")
	if err != nil {
		return nil, err
	}
	keysInfo, buf, err := ParseKeysInfo(result)
	if err != nil {
		return nil, err
	}
	result = io.NopCloser(bytes.NewReader(buf))
	c.keysInfo = keysInfo
	return result, nil
}

func (c *client) FetchInsecure(ctx context.Context) (io.ReadCloser, error) {
	if c.keysInfo == nil {
		return nil, fmt.Errorf("keys info struct not found. please fetch keys info")
	}
	return c.get(ctx, c.keysInfo.InsecureCAListURL)
}

func (c *client) FetchSecure(ctx context.Context) (io.ReadCloser, error) {
	if c.keysInfo == nil {
		return nil, fmt.Errorf("keys info struct not found. please fetch keys info")
	}
	return c.get(ctx, c.keysInfo.SecureCAListURL)
}

func (c *client) FetchSudo(ctx context.Context) (io.ReadCloser, error) {
	if c.keysInfo == nil {
		return nil, fmt.Errorf("keys info struct not found. please fetch keys info")
	}
	return c.get(ctx, c.keysInfo.SudoCAListURL)
}

func (c *client) FetchKRL(ctx context.Context) (io.ReadCloser, error) {
	if c.keysInfo == nil {
		return nil, fmt.Errorf("keys info struct not found. please fetch keys info")
	}
	r, err := c.get(ctx, c.keysInfo.KrlURL)
	if err != nil {
		return nil, err
	}
	decoder, err := zstd.NewReader(r)
	if err != nil {
		return nil, err
	}
	return decoder.IOReadCloser(), nil
}

func NewClient(cfg *config.CAuthConfig) *client {
	c := resty.New().
		SetBaseURL(cfg.URL).
		SetRetryCount(5).
		SetDoNotParseResponse(true).
		SetRedirectPolicy(resty.NoRedirectPolicy())
	if cfg.LegacySources != "" {
		c.SetQueryParam("sources", cfg.LegacySources)
	}

	keysClient := resty.New().
		SetRetryCount(5).
		SetDoNotParseResponse(true).
		SetRedirectPolicy(resty.NoRedirectPolicy())

	certPool, err := certifi.NewCertPool()
	if err == nil {
		c.SetTLSClientConfig(&tls.Config{RootCAs: certPool})
		keysClient.SetTLSClientConfig(&tls.Config{RootCAs: certPool})
	}

	return &client{
		c:          c,
		keysClient: keysClient,
	}
}
