package cauth

import (
	"bufio"
	"bytes"
	"context"
	"encoding/json"
	"io"
	"net/http"
	"net/url"
	"testing"

	"a.yandex-team.ru/infra/cauth/agent/linux/yandex-cauth-userd/internal/config"
	"github.com/klauspost/compress/zstd"
	"github.com/stretchr/testify/assert"
)

// TODO: use httpmock when it appear in contribs
// https://st.yandex-team.ru/CONTRIB-2332
type transportMock struct {
	host     string
	url      string
	response []byte
}

type readCloser struct {
	r io.Reader
}

func (r *readCloser) Read(p []byte) (n int, err error) {
	return r.r.Read(p)
}

func (r *readCloser) Close() error {
	return nil
}

func (t *transportMock) RoundTrip(request *http.Request) (*http.Response, error) {
	rsp := &http.Response{}
	if request.URL.Host != t.host || request.URL.Path != t.url {
		rsp.StatusCode = http.StatusNotFound
	} else {
		rsp.StatusCode = http.StatusOK
		rsp.Body = &readCloser{r: bytes.NewReader(t.response)}
	}
	return rsp, nil
}

func TestNewClient(t *testing.T) {
	cfg := config.CAuthConfig{
		URL:           "http://cauth-mock",
		LegacySources: "idm,cms,conductor,yp",
	}
	c := NewClient(&cfg)
	assert.NotNil(t, c)
	assert.Equal(t, "http://cauth-mock", c.c.HostURL)
	assert.Equal(t, "idm,cms,conductor,yp", c.c.QueryParam.Get("sources"))
	assert.Nil(t, c.keysInfo)

	cfg.LegacySources = ""
	c = NewClient(&cfg)
	assert.NotNil(t, c)
	assert.Equal(t, "http://cauth-mock", c.c.HostURL)
	assert.Equal(t, url.Values{}, c.c.QueryParam)
	assert.Nil(t, c.keysInfo)
}

func TestClient_FetchAccess(t *testing.T) {
	cfg := config.CAuthConfig{
		URL: "http://cauth-mock",
	}
	c := NewClient(&cfg)
	b := []byte("mock")
	c.c.GetClient().Transport = &transportMock{host: "cauth-mock", url: "/access/", response: b}
	rsp, err := c.FetchAccess(context.Background())
	assert.NoError(t, err)
	defer rsp.Close()
	rspBytes, err := io.ReadAll(rsp)
	assert.NoError(t, err)
	assert.Equal(t, b, rspBytes)
}

func TestClient_FetchAdminUsers(t *testing.T) {
	cfg := config.CAuthConfig{
		URL: "http://cauth-mock",
	}
	c := NewClient(&cfg)
	b := []byte("mock")
	c.c.GetClient().Transport = &transportMock{host: "cauth-mock", url: "/passwd/serveradmins/", response: b}
	rsp, err := c.FetchAdminUsers(context.Background())
	assert.NoError(t, err)
	defer rsp.Close()
	rspBytes, err := io.ReadAll(rsp)
	assert.NoError(t, err)
	assert.Equal(t, b, rspBytes)
}

func TestClient_FetchGroup(t *testing.T) {
	cfg := config.CAuthConfig{
		URL: "http://cauth-mock",
	}
	c := NewClient(&cfg)
	b := []byte("mock")
	c.c.GetClient().Transport = &transportMock{host: "cauth-mock", url: "/group/serverusers/", response: b}
	rsp, err := c.FetchGroup(context.Background())
	assert.NoError(t, err)
	defer rsp.Close()
	rspBytes, err := io.ReadAll(rsp)
	assert.NoError(t, err)
	assert.Equal(t, b, rspBytes)
}

func TestClient_FetchKeys(t *testing.T) {
	cfg := config.CAuthConfig{
		URL: "http://cauth-mock",
	}
	c := NewClient(&cfg)
	b := []byte("mock")
	c.c.GetClient().Transport = &transportMock{host: "cauth-mock", url: "/userkeys/", response: b}
	rsp, err := c.FetchKeys(context.Background())
	assert.NoError(t, err)
	defer rsp.Close()
	rspBytes, err := io.ReadAll(rsp)
	assert.NoError(t, err)
	assert.Equal(t, b, rspBytes)
}

func TestClient_FetchPasswd(t *testing.T) {
	cfg := config.CAuthConfig{
		URL: "http://cauth-mock",
	}
	c := NewClient(&cfg)
	b := []byte("mock")
	c.c.GetClient().Transport = &transportMock{host: "cauth-mock", url: "/passwd/serverusers/", response: b}
	rsp, err := c.FetchPasswd(context.Background())
	assert.NoError(t, err)
	defer rsp.Close()
	rspBytes, err := io.ReadAll(rsp)
	assert.NoError(t, err)
	assert.Equal(t, b, rspBytes)
}

func TestClient_FetchKeysInfo(t *testing.T) {
	cfg := config.CAuthConfig{
		URL: "http://cauth-mock",
	}
	c := NewClient(&cfg)
	ki := KeysInfo{
		InsecureCAListURL: "https://skotty.sec.yandex-team.ru/api/v1/ca/pub/insecure",
		SudoCAListURL:     "https://skotty.sec.yandex-team.ru/api/v1/ca/pub/sudo",
		KrlURL:            "https://skotty.sec.yandex-team.ru/api/v1/ca/krl/all.zst",
		SecureCAListURL:   "https://skotty.sec.yandex-team.ru/api/v1/ca/pub/secure",
		KeySources:        []string{"staff"},
	}
	b, err := json.Marshal(ki)
	assert.NoError(t, err)
	c.c.GetClient().Transport = &transportMock{host: "cauth-mock", url: "/keysinfo/", response: b}
	rsp, err0 := c.FetchKeysInfo(context.Background())
	assert.NoError(t, err0)
	rspki, _, err1 := ParseKeysInfo(rsp)
	assert.NoError(t, err1)
	assert.Equal(t, ki.InsecureCAListURL, rspki.InsecureCAListURL)
	assert.Equal(t, ki.SudoCAListURL, rspki.SudoCAListURL)
	assert.Equal(t, ki.KrlURL, rspki.KrlURL)
	assert.Equal(t, ki.SecureCAListURL, rspki.SecureCAListURL)
	assert.Equal(t, ki.KeySources, rspki.KeySources)
}

func TestClient_FetchInsecure(t *testing.T) {
	cfg := config.CAuthConfig{
		URL: "http://cauth-mock",
	}
	c := NewClient(&cfg)
	ki := KeysInfo{
		InsecureCAListURL: "https://skotty.sec.yandex-team.ru/api/v1/ca/pub/insecure",
		SudoCAListURL:     "https://skotty.sec.yandex-team.ru/api/v1/ca/pub/sudo",
		KrlURL:            "https://skotty.sec.yandex-team.ru/api/v1/ca/krl/all.zst",
		SecureCAListURL:   "https://skotty.sec.yandex-team.ru/api/v1/ca/pub/secure",
		KeySources:        []string{"staff"},
	}
	c.keysInfo = &ki
	b := []byte("mock")
	c.keysClient.GetClient().Transport = &transportMock{host: "skotty.sec.yandex-team.ru", url: "/api/v1/ca/pub/insecure", response: b}
	rsp, err := c.FetchInsecure(context.Background())
	assert.NoError(t, err)
	defer rsp.Close()
	rspBytes, err := io.ReadAll(rsp)
	assert.NoError(t, err)
	assert.Equal(t, b, rspBytes)
}

func TestClient_FetchSecure(t *testing.T) {
	cfg := config.CAuthConfig{
		URL: "http://cauth-mock",
	}
	c := NewClient(&cfg)
	ki := KeysInfo{
		InsecureCAListURL: "https://skotty.sec.yandex-team.ru/api/v1/ca/pub/insecure",
		SudoCAListURL:     "https://skotty.sec.yandex-team.ru/api/v1/ca/pub/sudo",
		KrlURL:            "https://skotty.sec.yandex-team.ru/api/v1/ca/krl/all.zst",
		SecureCAListURL:   "https://skotty.sec.yandex-team.ru/api/v1/ca/pub/secure",
		KeySources:        []string{"staff"},
	}
	c.keysInfo = &ki
	b := []byte("mock")
	c.keysClient.GetClient().Transport = &transportMock{host: "skotty.sec.yandex-team.ru", url: "/api/v1/ca/pub/secure", response: b}
	rsp, err := c.FetchSecure(context.Background())
	assert.NoError(t, err)
	defer rsp.Close()
	rspBytes, err := io.ReadAll(rsp)
	assert.NoError(t, err)
	assert.Equal(t, b, rspBytes)
}

func TestClient_FetchSudo(t *testing.T) {
	cfg := config.CAuthConfig{
		URL: "http://cauth-mock",
	}
	c := NewClient(&cfg)
	ki := KeysInfo{
		InsecureCAListURL: "https://skotty.sec.yandex-team.ru/api/v1/ca/pub/insecure",
		SudoCAListURL:     "https://skotty.sec.yandex-team.ru/api/v1/ca/pub/sudo",
		KrlURL:            "https://skotty.sec.yandex-team.ru/api/v1/ca/krl/all.zst",
		SecureCAListURL:   "https://skotty.sec.yandex-team.ru/api/v1/ca/pub/secure",
		KeySources:        []string{"staff"},
	}
	c.keysInfo = &ki
	b := []byte("mock")
	c.keysClient.GetClient().Transport = &transportMock{host: "skotty.sec.yandex-team.ru", url: "/api/v1/ca/pub/sudo", response: b}
	rsp, err := c.FetchSudo(context.Background())
	assert.NoError(t, err)
	defer rsp.Close()
	rspBytes, err := io.ReadAll(rsp)
	assert.NoError(t, err)
	assert.Equal(t, b, rspBytes)
}

func TestClient_FetchKRL(t *testing.T) {
	cfg := config.CAuthConfig{
		URL: "http://cauth-mock",
	}
	c := NewClient(&cfg)
	ki := KeysInfo{
		InsecureCAListURL: "https://skotty.sec.yandex-team.ru/api/v1/ca/pub/insecure",
		SudoCAListURL:     "https://skotty.sec.yandex-team.ru/api/v1/ca/pub/sudo",
		KrlURL:            "https://skotty.sec.yandex-team.ru/api/v1/ca/krl/all.zst",
		SecureCAListURL:   "https://skotty.sec.yandex-team.ru/api/v1/ca/pub/secure",
		KeySources:        []string{"staff"},
	}
	c.keysInfo = &ki
	b := []byte("mock")
	var buf bytes.Buffer
	wr := bufio.NewWriter(&buf)
	encoder, err0 := zstd.NewWriter(wr)
	assert.NoError(t, err0)
	var res []byte
	mockRsp := encoder.EncodeAll(b, res)
	c.keysClient.GetClient().Transport = &transportMock{host: "skotty.sec.yandex-team.ru", url: "/api/v1/ca/krl/all.zst", response: mockRsp}
	rsp, err := c.FetchKRL(context.Background())
	assert.NoError(t, err)
	defer rsp.Close()
	rspBytes, err := io.ReadAll(rsp)
	assert.NoError(t, err)
	assert.Equal(t, b, rspBytes)
}
